<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PendingBulkUploadRequestsResource\Pages;
use App\Filament\Resources\PendingBulkUploadRequestsResource\RelationManagers;
use App\Models\BulkUpload;
use App\Models\PendingBulkUploadRequests;
use App\Models\WithdrawRequest;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;

class PendingBulkUploadRequestsResource extends Resource
{
    protected static ?string $model = BulkUpload::class;

    protected static ?string $navigationGroup  = "Sites";

    protected static ?string $label = "Pending Bulk Upload Requests";

    protected static ?string $navigationIcon = 'heroicon-o-book-open';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                BulkUpload::with("user")->whereIn('status',[0,1])
            )
            ->defaultSort("created_at","desc")
            ->recordUrl(null)
            ->paginated(config("app.pagination"))
            ->columns([
                TextColumn::make('vendor.name')->label("Name")->sortable()->wrap(),
                TextColumn::make('user.email')->label("Email")->sortable()->wrap(),

                TextColumn::make("created_at")->sortable()->label("Datetime")->dateTime(),
                TextColumn::make("status")->badge()->formatStateUsing(function (BulkUpload $record) {
                    if($record->status == 0){
                        return 'No Excel File Upload';
                    }else if($record->status == 1){
                        return 'Excel File Uploaded. Waiting Blogger to accept/decline';
                    }
                })->color(function (BulkUpload $record) {
                    if($record->status == 0){
                        return "danger";
                    }else if($record->status == 1){
                        return "warning";
                    }
                }),
                TextColumn::make("id")->label("Excel File")
                    ->icon("heroicon-o-arrow-down-tray")->formatStateUsing(function (BulkUpload $record){
                    return new HtmlString('&nbsp;');
                })->tooltip("Download Excel File")->url(function (BulkUpload $record) {
                        $file = Str::startsWith($record->filename,"uploads/") ? "/".$record->filename : "/uploads/".$record->filename;
                    return url($file);
                })
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                Filter::make("vendor")->form([
                    TextInput::make("name"),
                    TextInput::make("email"),

                ])->columns(6)->columnSpanFull()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['name'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('name', 'LIKE','%'.$data['name'].'%');
                                });
                            })
                            ->when($data['email'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('email', 'LIKE','%'.$data['email'].'%');
                                });
                            });
                    }),
            ])
            ->actions([
                Tables\Actions\Action::make("upload_excel_file")->label(function (BulkUpload $record){
                    if( $record->status == 0 ){
                        return 'Upload Excel File';
                    }
                    elseif ($record->status == 1){
                        return 'View Uploaded Sites';
                    }
                })->url(function (BulkUpload $record) {
                    if($record->status == 0){
                        return PendingBulkUploadRequestsResource::getUrl('upload-excel-file', ['record' => $record->id]);
                    }

                    if ($record->status == 1) {
                        return PendingBulkUploadRequestsResource::getUrl('bulk-uploaded-sites', ['record' => $record->id]);
                    }
                })
            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPendingBulkUploadRequests::route('/'),
            'create' => Pages\CreatePendingBulkUploadRequests::route('/create'),
            'edit' => Pages\EditPendingBulkUploadRequests::route('/{record}/edit'),
            'upload-excel-file' => Pages\UploadExcelFile::route("/{record}/upload-excel-file"),
            'bulk-uploaded-sites'   =>  Pages\ViewBulkUploadedSites::route("/{record}/bulk-uploaded-sites"),
        ];
    }
}
