<?php

namespace App\Filament\Resources;

use App\Filament\Exports\BloggerExportExporter;
use App\Filament\Pages\BloggersPerformance;
use App\Filament\Pages\CompletedOrders;
use App\Filament\Pages\PendingOrders;
use App\Filament\Pages\WalletHistory;
use App\Filament\Resources\BloggersListResource\Pages;
use App\Filament\Resources\BloggersListResource\RelationManagers;
use App\Models\BloggersList;
use App\Models\NewOrderProcessDetail;
use App\Models\User;
use App\Project\NewOrderStatus;
use Carbon\Carbon;
use Filament\Actions\Exports\Enums\ExportFormat;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\HtmlString;
use Maatwebsite\Excel\Excel;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class BloggersListResource extends Resource
{
    protected static ?string $model = User::class;
    protected static ?string $label = "Bloggers List";



    protected static ?string $navigationIcon = 'heroicon-o-user-group';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                User::whereIn('role',['vendor'])
            )
            ->recordUrl(null)
            ->defaultSort("created_at","desc")
            ->selectCurrentPageOnly(true)
            ->paginated(config('app.pagination'))
            ->columns([
                Tables\Columns\TextColumn::make("name")->sortable()->wrap(),
                Tables\Columns\TextColumn::make("email")->sortable()->wrap(),

                Tables\Columns\TextColumn::make("status")->toggleable()->label("Account Status")->formatStateUsing(function (User $record) {
                    return $record->status ? "Active" : "Blocked";
                })->badge()
                    ->color(function (User $record) {
                        if($record->status == 1){
                            return "success";
                        }else{
                            return "danger";
                        }
                    })
                    ->action(function (User $record) {
                        if( $record->status == 1 ){
                            $record->status = 0;
                        }else{
                            $record->status = 1;
                        }
                        $record->save();
                    }),
                Tables\Columns\TextColumn::make("id")->toggleable()->label("Total Orders")->formatStateUsing(function (User $record) {
                    $pending_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER,NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER])->count();
                    $completed_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED])->count();
                    return $pending_orders + $completed_orders;
                }),
                Tables\Columns\TextColumn::make("role")->toggleable()->label("Pending Orders")->formatStateUsing(function (User $record) {
                    $pending_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER,NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER])->count();
                    return new HtmlString('<span style="text-decoration: underline">'.$pending_orders.'</span>') ;
                })->url(function (User $record) {
                    return PendingOrders::getUrl(["id"=>$record->id]);
                }),
                Tables\Columns\TextColumn::make("updated_at")->toggleable()->label("Completed Orders")->formatStateUsing(function (User $record) {
                    $completed_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED])->count();
                    return new HtmlString('<span style="text-decoration: underline">'.$completed_orders.'</span>') ;
                })->url(function (User $record) {
                    return CompletedOrders::getUrl(["id"=>$record->id]);
                }),

                Tables\Columns\TextColumn::make("created_at")->toggleable()->label("Wallet")->formatStateUsing(function (User $record) {
                    return new HtmlString('<span style="text-decoration: underline">'.admin_vendor_balance($record->id).'</span>') ;
                })->url(function (User $record) {
                    return WalletHistory::getUrl(["id"=>$record->id]);
                }),
                Tables\Columns\TextColumn::make("current_login")->toggleable()->label("Last Login")->formatStateUsing(function (User $record) {
                    return $record->current_login ? Carbon::parse($record->current_login)->toDayDateTimeString() : "";
                })->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make("login_count")->toggleable()->label("Login Counts")->formatStateUsing(function (User $record) {
                    return $record->login_count ?? 0;
                }),


            ])
            ->bulkActions([

                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForWriterType()->askForFilename()->withColumns([
                        Column::make("name"),
                        Column::make("email"),
                        Column::make("status")->heading("Account Status")->formatStateUsing(function ($record){
                            return $record->status ? "Active" : "Blocked";
                        }),
                        Column::make("id")->heading("Total Orders")->formatStateUsing(function ($record) {
                            $pending_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER,NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER])->count();
                            $completed_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED])->count();
                            return $pending_orders + $completed_orders;
                        }),
                        Column::make("role")->heading("Pending Orders")->formatStateUsing(function ($record) {
                            $pending_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER,NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER])->count();
                            return $pending_orders ;
                        }),
                        Column::make("updated_at")->heading("Completed Orders")->formatStateUsing(function ($record) {
                            $completed_orders= NewOrderProcessDetail::where('vendor_id',$record->id)->whereIn('status',[NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED])->count();
                            return $completed_orders ;
                        }),

                        Column::make("created_at")->heading("Wallet")->formatStateUsing(function ( $record) {
                            return admin_vendor_balance($record->id);
                        }),
                        Column::make("current_login")->heading("Last Login")->formatStateUsing(function ( $record) {
                            return $record->current_login ? Carbon::parse($record->current_login)->toDayDateTimeString() : "";
                        }),
                        Column::make("login_count")->heading("Login Counts")->formatStateUsing(function ( $record) {
                            return $record->login_count ?? 0;
                        }),
                    ])
                ])


            ])
            ->filters([
                Tables\Filters\Filter::make("search")->form([
                    Forms\Components\TextInput::make("name"),
                    Forms\Components\TextInput::make("email"),
                ])->columns(6)->columnSpanFull()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['name'], function (Builder $query) use ($data) {
                                return $query->where('name', 'LIKE','%'.$data['name'].'%');
                            })
                            ->when($data['email'], function (Builder $query) use ($data) {
                                return $query->where('email', 'LIKE','%'.$data['email'].'%');
                            })
                            ;
                    }),
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->actions([
                Tables\Actions\Action::make("reset_password")
                    ->icon("heroicon-o-arrow-path-rounded-square")
                    ->label(" ")
                    ->tooltip("Reset Password to 12345678")
                    ->action(function (User $record) {
                        $record->password = bcrypt("12345678");
                        $record->save();
                        Notification::make("success")->title("Password Reset")->send();
                    })
                ,
                Tables\Actions\Action::make("performance")
                    ->icon("heroicon-o-chart-bar")
                    ->label(" ")
                    ->tooltip("Performance")
                    ->url(function (User $record) {
                        return BloggersPerformance::getUrl(["id"=>$record->id]);
                    })

            ])
           ;
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBloggersLists::route('/'),
            'create' => Pages\CreateBloggersList::route('/create'),
            'edit' => Pages\EditBloggersList::route('/{record}/edit'),
        ];
    }
}
