<?php

namespace App\Filament\Pages;

use App\Exports\WithdrawRequestExport;
use App\Models\WalletHistory;
use App\Models\WithdrawRequest;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Pages\Page;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Maatwebsite\Excel\Facades\Excel;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class WithdrawalRequests extends Page implements HasForms, HasTable
{
    use InteractsWithTable;
    use InteractsWithForms;
    protected static ?string $navigationGroup = "Wallet";
    protected static ?string $navigationIcon = 'heroicon-o-receipt-refund';

    protected static string $view = 'filament.pages.withdrawal-requests';
    public function table(Table $table): Table
    {
        return $table
            ->query(
                WithdrawRequest::has("wallet_histories")->withSum(['wallet_histories as total_price' => function ($query) {
                $query->select(DB::raw('SUM(price)'));
            }], 'price')->where('status',0)
                    )
            ->defaultSort('created_at','desc')
            ->headerActions([
                Action::make("export")->label("Download Bank Details")->action(function (){
                    return Excel::download(new WithdrawRequestExport, Carbon::now().'withdraw_request.xlsx');
                })
            ])
            ->columns([

                TextColumn::make('user.name')->description(function (WithdrawRequest $record) {
                    return $record->user->email;
                })->wrap(),
                TextColumn::make("created_at")->label("Payment Method")
                    ->wrap()
                    ->description(function (WithdrawRequest $record) {
                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return new HtmlString('
                                <strong>Bank Type:- </strong>'.$record->wallet_histories[0]->bank_type.' <br>
                                <strong>Beneficiary Account Number:- </strong>'.$record->wallet_histories[0]->beneficiary_account_number.' <br>
                                <strong>Beneficiary Name:- </strong>'.$record->wallet_histories[0]->beneficiary_name.' <br>
                                <strong>Customer Reference Number:- </strong>'.$record->wallet_histories[0]->customer_reference_number.' <br>
                                <strong>Beneficiary Bank Name:- </strong>'.$record->wallet_histories[0]->bene_bank_name.' <br>
                                <strong>Beneficiary Bank Branch Name:- </strong>'.$record->wallet_histories[0]->bene_bank_branch_name.' <br>
                                <strong>IFSC Code:- </strong>'.$record->wallet_histories[0]->ifsc_code.' <br>
                                <strong>Beneficiary Email id:- </strong>'.$record->wallet_histories[0]->beneficiary_email_id .' <br>'

                            );
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
                            return new HtmlString('<div class="flex items-center flex-col justify-between border-b border-gray-200 mt-2 pb-1 mb-1" >

                                    <span class="text-black font-bold" >
                                    <a href = "/uploads/'.$record->wallet_histories[0]->qr_code_image .'" target = "_blank" >
                                        <img src = "/uploads/'.$record->wallet_histories[0]->qr_code_image .'" class="w-32" >
                                    </a >
                                    </span >
                                </div >');
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return $record->wallet_histories[0]->paypal_email;

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return $record->wallet_histories[0]->upi_id;

                        }

                    })
                    ->badge()
                    ->color(function (WithdrawRequest $record) {
                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return 'success';
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
                            return 'danger';
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return 'info';

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return 'warning';

                        }

                    })
                    ->formatStateUsing(function (WithdrawRequest $record) {

                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return 'Bank Details';
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
                            return 'QR Code';
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return 'Paypal ID';

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return 'UPI ID';

                        }



                    }),
                TextColumn::make("invoice_number")->formatStateUsing(function (WithdrawRequest $record) {
                    return $record->invoice_pre.$record->invoice_number;
                }),
                TextColumn::make("total_price")->label("Amount")->summarize(Sum::make())->sortable(),



                TextColumn::make("updated_at")->label("Datetime")->formatStateUsing(function (WithdrawRequest $record) {
                    return new HtmlString(Carbon::parse($record->created_at)->toDayDateTimeString() );
                }),

            ])
            ->paginated(config('app.pagination'))
            ->filtersLayout(FiltersLayout::AboveContentCollapsible)
            ->filters([
                Filter::make("search")->form([
                    TextInput::make("name"),
                    TextInput::make("email"),
                    Select::make("payment_method")->options([
                        "bank" => "Bank",
                        "qr_code" => "QR Code",
                        "paypal_id" => "Paypal ID",
                        "upi_id" => "UPI ID",
                    ])
                ])->columns(6)->columnSpanFull()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['name'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('name', 'LIKE','%'.$data['name'].'%');
                                });
                            })
                            ->when($data['email'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('email', 'LIKE','%'.$data['email'].'%');
                                });
                            })

                            ->when($data['payment_method'], function (Builder $query) use ($data) {
                                return $query->whereHas("wallet_histories", function (Builder $query) use ($data) {
                                    return $query->where('payment_method', $data['payment_method']);
                                });
                            });
                    }),
                DateRangeFilter::make("created_at")->label("Date")
            ])
            ->actions([
                Action::make("detail")
                    ->label(" ")
                    ->icon("heroicon-s-eye")
                    ->tooltip("Detail")
                    ->modalContent(function (WithdrawRequest $record) {
                    return view("welcome")->with("record", $record->id)->with("action", "new");
                })->modalSubmitAction(false)->modalCancelAction(false)
            ])
            ->bulkActions([

            ]);
    }
}
