<?php

namespace App\Filament\Manager\Resources;

use App\Filament\Manager\Resources\PendingApprovalForWriterResource\Pages;
use App\Filament\Manager\Resources\PendingApprovalForWriterResource\RelationManagers;
use App\Models\NewOrder;
use App\Models\NewOrderProcess;
use App\Models\NewOrderProcessDetail;
use App\Models\PendingApprovalForWriter;
use App\Models\User;
use App\Project\NewOrderStatus;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class PendingApprovalForWriterResource extends Resource
{
    protected static ?string $model = NewOrder::class;

    protected static ?string $label = "Pending Approval For Writer";

    protected static ?string $navigationLabel = "Writers";

    protected static ?string $navigationGroup = "Pending Approval";

    protected static ?string $navigationIcon = 'heroicon-o-building-office-2';



    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function getNavigationBadge(): ?string
    {
        return manager_pending_approvals_for_writer();
    }


    public static function table(Table $table): Table
    {
        return $table
            ->query(
                NewOrder::with("order_processes_writer_pushed_to_manager.new_order_process_details")
                    ->where('manager_id',Auth::id())
                    ->where('new_order_status',NewOrderStatus::$WRITER_PUSHED_TO_MANAGER)
            )
            ->recordUrl(null)
            ->columns([
                Tables\Columns\TextColumn::make("order_id")->label("Order Id")->sortable()->searchable(),
                Tables\Columns\TextColumn::make("order_processes_writer_pushed_to_manager.writer.name")->toggleable()->sortable()->description(function (NewOrder $record) {
                    return $record->order_processes_writer_pushed_to_manager->writer->email;
                }),
                Tables\Columns\TextColumn::make("client_name")->label("Client Name")->toggleable()->searchable()->sortable(),
                Tables\Columns\TextColumn::make("client_website")
                    ->toggleable()
                    ->copyable()
                    ->icon("heroicon-o-document-duplicate")
                    ->tooltip("Click here to Copy")
                    ->label("Client Website")->searchable()->sortable(),
                Tables\Columns\TextColumn::make("no_of_links")->label("No of Links")->toggleable()->sortable(),
                Tables\Columns\TextColumn::make("order_type")->label("Order Type")->toggleable()->badge()->sortable()->color(function (NewOrder $record) {
                    if($record->order_type == "gp"){
                        return 'info';
                    }else{
                        return 'success';
                    }
                }),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\Action::make("detail")
                    ->icon("heroicon-o-eye")
                    ->label(" ")
                    ->color("success")
                    ->tooltip("Detail")
                    ->url(function (NewOrder $record) {
                   return  PendingApprovalForWriterResource::getUrl('detail', ['record' => $record->id]);
                }),
                Tables\Actions\Action::make("post_disapproved")
                    ->icon("heroicon-o-trash")
                    ->label(" ")
                    ->color("danger")
                    ->tooltip("Post Disapproved")
                    ->form([
                    Forms\Components\Textarea::make("note")->label("Note")
                ])->action(function (NewOrder $record, array $data) {
                    $order = NewOrder::with("order_processes_writer_pushed_to_manager.new_order_process_details")
                        ->where('manager_id', Auth::id())
                        ->where('new_order_status', NewOrderStatus::$WRITER_PUSHED_TO_MANAGER)
                        ->where('id', $record->id)->firstorFail();

                    $status= NewOrderStatus::$MANAGER_REJECTED_AND_PUSHED_BACK_TO_WRITER;

                    try{
                        DB::beginTransaction();
                        $order->new_order_status = $status;
                        $order->save();

                        $new_order_process= new NewOrderProcess();
                        $new_order_process->new_order_id= $order->id;
                        $new_order_process->writer_id= $order->order_processes_writer_pushed_to_manager->writer_id;
                        $new_order_process->manager_id= Auth::id();
                        $new_order_process->note = $data['note'];
                        $new_order_process->status = $status;
                        $new_order_process->save();

                        foreach ($order->order_processes_writer_pushed_to_manager->new_order_process_details as $det) {
                            $detail = new NewOrderProcessDetail();
                            $detail->new_order_process_id = $new_order_process->id;
                            $detail->new_site_id = $det->new_site_id;
                            $detail->url = $det->url;
                            $detail->price = $det->price;
                            $detail->anchor = $det->anchor;
                            $detail->title = $det->title;
                            $detail->ourl = $det->ourl;
                            $detail->doc_urls = $det->doc_urls;
                            $detail->upload_doc_file = $det->upload_doc_file;
                            $detail->insert_after = $det->insert_after;
                            $detail->statement = $det->statement;
                            $detail->note = $det->note;
                            $detail->vendor_id = $det->new_site->uploaded_user_id;
                            $detail->status = $status;
                            $detail->save();

                        }

                        DB::commit();
                        Notification::make("success")->title("Pushed Back to Writer")->send();

                        $writer= User::where('role','writer')->where('id',$order->order_processes_writer_pushed_to_manager->writer_id)->first();
                        $admin = User::where('role','admin')->first();

                        Notification::make()
                            ->title("Order Rejected by ".Auth::user()->name.'('.Auth::user()->email.')')
                            ->body("Order No: ".$record->order_id)
                            ->success()
                            ->sendToDatabase([$writer,$admin]);
                    }catch (\Exception $exception){
                        DB::rollBack();
                        dd($exception->getMessage());
                    }
                })
            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->withColumns([
                        Column::make("order_id")->heading("Order Id"),
                        Column::make("order_processes_writer_pushed_to_manager.writer.name")->heading("Writer Name"),
                        Column::make("order_processes_writer_pushed_to_manager.writer.email")->heading("Writer Email"),
                        Column::make("client_name")->heading("Client Name"),
                        Column::make("client_website")->heading("Client Website"),
                        Column::make("no_of_links")->heading("No of Links"),
                        Column::make("order_type")->heading("Order Type"),

                    ])
                ])
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPendingApprovalForWriters::route('/'),
            'create' => Pages\CreatePendingApprovalForWriter::route('/create'),
            'edit' => Pages\EditPendingApprovalForWriter::route('/{record}/edit'),
            'detail' => Pages\DetailPage::route('/{record}'),
        ];
    }
}
