<?php

namespace App\Filament\Clusters\Reporting\Pages;

use App\Filament\Clusters\Reporting;
use Carbon\Carbon;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Pages\Page;
use Filament\Tables\Columns\Summarizers\Summarizer;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class PriceReportingPage extends Page implements HasForms,HasTable
{
    use InteractsWithTable;
    use InteractsWithForms;

    protected static ?string $navigationGroup = "Reports";
    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.clusters.reporting.pages.price-reporting-page';

    protected static ?string $cluster = Reporting::class;

    public function table(Table $table): Table
    {
        return $table
            ->query(function (){

                return \App\Models\WalletHistory::selectRaw("wallet_histories.id as id, wallet_histories.price as amount,count(wallet_histories.price) as count")
                    ->join("withdraw_requests", "withdraw_requests.id", "wallet_histories.withdraw_request_id")
                    ->where("withdraw_requests.status", 1)
                    ->groupBy(["wallet_histories.price"]);

            })

            ->paginated(config("app.pagination"))
            ->columns([
                TextColumn::make("amount")->formatStateUsing(function ($record){
                    return $record->amount;
                }),
                TextColumn::make("count")->formatStateUsing(function ($record){
                    return $record->count;
                }),
                TextColumn::make("id")->label("Total")->formatStateUsing(function ($record){
                    return $record->count * $record->amount;
                })->summarize(
                    Summarizer::make()
                        ->label("Total Amount")
                        ->using(function($query) {
                            $data = $query->get();
                            $total = 0;

                            foreach ($data as $record){
                                $total += $record->count * $record->amount;
                            }

                            return $total;
                        })
                ),
            ])

            ->filtersLayout(FiltersLayout::AboveContent)
            ->filters([
                Filter::make("search")->form([

                    DatePicker::make("start_date")->default(function (){
                        return Carbon::today()->subMonth()->format("Y-m-d");
                    })->label("Start Date"),

                    DatePicker::make("end_date")
                        ->default(function (){
                            return Carbon::today()->format("Y-m-d");
                        })
                        ->label("End Date"),

                ])
                    ->query(function (Builder $query, array $data){
                        return $query->when($data['start_date'] && $data['end_date'], function (Builder $query) use ($data){
                            $query->whereBetween('wallet_histories.approved_date', [$data["start_date"].' 00:00:00', $data["end_date"].' 23:59:59' ]);
                        });

                    })
                    ->columns(4)->columnSpanFull()
            ])
            ->actions([
                // ...
            ])
            ->bulkActions([
                // ...
            ]);
    }
}
