<?php

namespace App\Filament\Clusters\Reporting\Pages;

use App\Filament\Clusters\Reporting;
use App\Models\OrdersView;
use Carbon\Carbon;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Pages\Page;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Grouping\Group;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class ClientReportPage extends  Page implements HasForms,HasTable
{
    use InteractsWithTable;
    use InteractsWithForms;

    protected static ?string $navigationLabel = "Client Report";

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.clusters.reporting.pages.client-report-page';

    protected static ?string $cluster = Reporting::class;


    public function table(Table $table): Table
    {
        return $table
            ->query(function (){

                return OrdersView::selectRaw("orders_view.id,orders_view.client_website,orders_view.client_name,orders_view.order_id,orders_view.created_at,orders_view.new_order_status,orders_view.no_of_links,new_sites.root_domain,new_order_process_details.price")
                    ->leftJoin("users","users.id","orders_view.manager_id")
                    ->leftJoin("new_order_processes","new_order_processes.new_order_id","orders_view.id")
                    ->leftJoin("new_order_process_details","new_order_process_details.new_order_process_id","new_order_processes.id")
                    ->leftJoin("new_sites","new_sites.id","new_order_process_details.new_site_id")
                    ->whereRaw('total = total1 and total > 0')
                    ->groupByRaw("orders_view.id,orders_view.client_website,orders_view.client_name,orders_view.order_id,orders_view.created_at,orders_view.new_order_status,orders_view.no_of_links,new_sites.root_domain,new_order_process_details.price");

            })
            ->defaultGroup("order_id")
            ->groups([
                Group::make("order_id")->label("Order Id")->collapsible()
            ])
            ->paginated(config("app.pagination"))
            ->columns([
                TextColumn::make("client_name")->formatStateUsing(function ($record){
                    return $record->client_name;
                }),
                TextColumn::make("client_website")->formatStateUsing(function ($record){
                    return $record->client_website;
                }),
                TextColumn::make("order_id")->formatStateUsing(function ($record){
                    return $record->order_id;
                }),
                TextColumn::make("root_domain")->formatStateUsing(function ($record){
                    return $record->root_domain;
                }),
                TextColumn::make("price")
                    ->formatStateUsing(function ($record){
                        return $record->price;
                    })
                    ->summarize(Sum::make()->label("Total Amount")),

            ])

            ->actions([
                // ...
            ])
            ->filtersLayout(FiltersLayout::AboveContent)
            ->filters([
                Filter::make("search")->form([

                    TextInput::make("order_id")->label("Order Id"),
                    DatePicker::make("start_date")->default(function (){
                        return Carbon::today()->subMonth()->format("Y-m-d");
                    })->label("Start Date"),

                    DatePicker::make("end_date")
                        ->default(function (){
                            return Carbon::today()->format("Y-m-d");
                        })
                        ->label("End Date"),

                ])
                    ->query(function (\Illuminate\Database\Eloquent\Builder $query, array $data){
                        return
                            $query
                                ->when($data['start_date'], function (Builder $query) use ($data){
                                    $query->whereDate('orders_view.created_at', '>=', $data['start_date']);
                                })
                                ->when($data['end_date'], function (Builder $query) use ($data){
                                    $query->whereDate('orders_view.created_at', '<=', $data['end_date']);
                                })
                                ->when($data['order_id'], function (Builder $query) use ($data){
                                    $query->where('orders_view.order_id', 'LIKE','%'.$data['order_id'].'%');
                                })
                            ;

                    })
                    ->columns(4)->columnSpanFull()
            ])
            ->bulkActions([
                // ...
            ]);
    }
}
