<?php

namespace App\Filament\Blogger\Resources;

use App\Filament\Blogger\Resources\ViewAllSitesResource\Pages;
use App\Filament\Blogger\Resources\ViewAllSitesResource\RelationManagers;
use App\Models\NewOrder;
use App\Models\NewSite;
use App\Models\ViewAllSites;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Infolists\Components\TextEntry;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\HtmlString;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;
use Tapp\FilamentValueRangeFilter\Filters\ValueRangeFilter;

class ViewAllSitesResource extends Resource
{
    protected static ?string $model = NewOrder::class;
    protected static ?string $label = "View All Sites";

    protected static ?string $navigationGroup = "Sites";


    protected static ?string $navigationIcon = 'heroicon-o-globe-alt';


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->striped()
            ->recordUrl(null)
            ->query(
                NewSite::where('bulk_upload_status',null)
                    ->where('bulk_upload_id',null)
                    ->where('delete_site',null)
                    ->where('uploaded_user_id',Auth::id())
            )
            ->paginated(config('app.pagination'))
            ->columns([
                Tables\Columns\TextColumn::make('root_domain')->label("Root Domain")
                    ->formatStateUsing(function (NewSite $record){
                        if(Carbon::parse($record->created_at)->diffInDays(Carbon::now()) <= 30){
                            $text = "<span style='background: green;font-size: 11px;color: white;display: inline-block;margin-left: 3px;padding: 1px 7px;'>New Arrival</span>";
                        }else{
                            $text =  '';
                        }

                        return new HtmlString($record->root_domain. $text);
                    })->wrap()
                    ->description(function (NewSite $record) {
                        $text ='';
                        $text .= 'Category:- '.$record->category.'<br>';
                        $text .= 'Added on:- '.$record->created_at->toDayDateTimeString().'<br>';


                        return new HtmlString('<span style="font-size: 12px;">'.$text.'</span>');
                    }),



                Tables\Columns\TextColumn::make('dr')->toggleable()->label("DR")->sortable(),
                Tables\Columns\TextColumn::make('da')->toggleable()->label("DA")->sortable(),
                Tables\Columns\TextColumn::make('rd')->toggleable()->label("RD")->sortable(),
                Tables\Columns\TextColumn::make('spam_score')->toggleable()->sortable(),

                Tables\Columns\TextColumn::make('traffic_source')->toggleable()->label("Traffic")->sortable(),
                Tables\Columns\TextColumn::make('gp_price')->toggleable()->label("GP Price")->sortable(),
                Tables\Columns\TextColumn::make('niche_edit_price')->toggleable()->label("Niche Price")->sortable(),
                Tables\Columns\TextColumn::make('fc_gp')->toggleable()->label("FC GP")->sortable(),
                Tables\Columns\TextColumn::make('fc_ne')->toggleable()->label("FC NE")->sortable(),
                Tables\Columns\TextColumn::make('website_status')->toggleable()->label("Website Status")->badge()->color(function (NewSite $d){
                    if($d->website_status == 'Acceptable'){
                        return 'success';
                    }
                    if($d->website_status == 'Non Acceptable'){
                        return 'danger';
                    }
                })->sortable(),
                Tables\Columns\TextColumn::make('email')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(function (NewSite $d) {
                        if(!empty($d->email)) {
                            return $d->email;
                        }else{
                            return optional($d->uploaded_user)->email;
                        }
                    }),

            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                Tables\Filters\Filter::make("Filters")->form([
                    Forms\Components\TextInput::make("category"),
                    Forms\Components\TextInput::make("root_domain"),
                    Forms\Components\TextInput::make("website_niche"),
                    Forms\Components\Select::make("site_status")->live()->label("Status")
                        ->options([
                            "0"    =>  "Pending",
                            "1"    =>  "Approved",
                            "2"    =>  "Rejected",
                        ]),
                ])->columns(6)->columnSpanFull()->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when($data['category'], function (Builder $query) use ($data) {
                            return $query->where('category', 'LIKE','%'.$data['category'].'%');
                        })
                        ->when($data['root_domain'], function (Builder $query) use ($data) {
                            return $query->where('root_domain', 'LIKE','%'.$data['root_domain'].'%');
                        })
                        ->when($data['site_status'] != "", function (Builder $query) use ($data) {
                            return $query->where('site_status', $data['site_status']);
                        })
                        ->when($data['website_niche'], function (Builder $query) use ($data) {
                            return $query->where('website_niche', 'LIKE','%'.$data['website_niche'].'%');
                        });
                }),
                TernaryFilter::make('new_arrival')
                    ->label('New Arrival')
                    ->queries(
                        true: fn (Builder $query) => $query->where('created_at','>=',Carbon::now()->subDays(30)),
                        false: fn (Builder $query) => $query->where('created_at','<',Carbon::now()->subDays(30)),
                        blank: fn (Builder $query) => $query, // In this example, we do not want to filter the query when it is blank.
                    ),
                DateRangeFilter::make("created_at")->label("Added on"),
                ValueRangeFilter::make("dr")->label("DR"),
                ValueRangeFilter::make("da")->label("DA"),
                ValueRangeFilter::make("rd")->label("RD"),
                ValueRangeFilter::make("traffic_source")->label("Traffic"),
                ValueRangeFilter::make("gp_price"),
                ValueRangeFilter::make("niche_edit_price"),

            ])
            ->actions([
                Tables\Actions\Action::make("view")->icon("heroicon-m-eye")->label(" ")->tooltip("view")->infolist([
                    TextEntry::make("website_niche")->inlineLabel(),
                    TextEntry::make("sample_url")->inlineLabel(),
                    TextEntry::make("href_url")->formatStateUsing(function (NewSite $d) {
                        return new HtmlString('<span style="word-wrap: anywhere; ">'.$d->href_url.'</span>');
                    }),
                    TextEntry::make("paypal_id")->inlineLabel(),
                    TextEntry::make("skype")->inlineLabel(),
                    TextEntry::make("whatsapp")->inlineLabel(),
                    TextEntry::make("country_source")->inlineLabel(),


                ]),
            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForWriterType()->askForFilename()->withColumns([
                        Column::make("root_domain"),
                        Column::make("category"),
                        Column::make("created_at")->heading('Added on'),
                        Column::make("site_status")->formatStateUsing(function ($state){
                            if($state == 0){
                                return 'Pending';
                            }elseif($state == 2) {
                                return 'Rejected';
                            }
                            elseif( $state == 1 ) {
                                return 'Approved';
                            }
                        }),
                        Column::make("dr"),
                        Column::make("da"),
                        Column::make("spam_score"),
                        Column::make("rd"),
                        Column::make("traffic_source"),
                        Column::make("gp_price"),
                        Column::make("niche_edit_price"),
                        Column::make("fc_gp"),
                        Column::make("fc_ne"),
                        Column::make("website_status"),
                        Column::make("email"),
                        Column::make("website_niche"),
                        Column::make("sample_url"),
                        Column::make("href_url"),
                        Column::make("paypal_id"),
                        Column::make("skype"),
                        Column::make("whatsapp"),
                        Column::make("country_source"),

                    ])
                ])
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListViewAllSites::route('/'),
            'create' => Pages\CreateViewAllSites::route('/create'),
            'edit' => Pages\EditViewAllSites::route('/{record}/edit'),
        ];
    }
}
