<?php

namespace App\Filament\Blogger\Resources;

use App\Filament\Blogger\Resources\OrderResource\Pages;
use App\Filament\Blogger\Resources\OrderResource\RelationManagers;
use App\Harman\Helpers\NewLinkVerify;
use App\Models\NewOrderProcessDetail;
use App\Models\Order;
use App\Models\User;
use App\Project\NewOrderStatus;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use Nette\Utils\Html;

class OrderResource extends Resource
{
    protected static ?string $model = NewOrderProcessDetail::class;
    protected static ?string $label = "Orders";

    protected static ?string $navigationGroup = "Orders";

    protected static ?string $navigationIcon = 'heroicon-o-credit-card';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                NewOrderProcessDetail::where('vendor_id',Auth::id())
                    ->whereIn('status',[
                            NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER,
                            NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS,
                            NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER,
                            NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED,
                            NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION
                    ])
            )
            ->defaultSort('updated_at','desc')
            ->paginated(config("app.pagination"))
            ->recordUrl(null)
            ->striped()
            ->columns([
                Tables\Columns\TextColumn::make("new_order_process.new_order.order_id")->label("Order ID")->badge()->sortable()->label("Order Id"),
                Tables\Columns\TextColumn::make("new_site.root_domain")->label("Root Domain"),
                Tables\Columns\TextColumn::make("created_at")->toggleable()->label("Date/Time"),
                Tables\Columns\TextColumn::make("status")->toggleable()->formatStateUsing(function (NewOrderProcessDetail $record){
                    $detail = $record;
                    if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return new HtmlString('Completed');
                    if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return new HtmlString('Blogger Rejected the Order');
                    if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return 'Waiting From Manager';
                    if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return new HtmlString('Pending Order');
                    if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return 'Rejected by Manager';
                    return '';
                })->badge(),
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                DateRangeFilter::make("created_at")->label("Search By Date"),
                Tables\Filters\Filter::make("Search")->form([
                    TextInput::make("order_id")->label("Order Id"),
                    TextInput::make("root_domain")->label("Search by Root Domain"),
                ])->query(function (Builder $query,$data){
                    return $query
                        ->when($data['order_id'],function (Builder $query) use ($data) {
                            $query->whereHas("new_order_process.new_order",function (Builder $query) use ($data){
                                $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            });
                        })
                        ->when($data['root_domain'],function (Builder $query) use ($data) {
                            $query->whereHas("new_site",function (Builder $query) use ($data){
                                $query->where('root_domain','LIKE','%'.$data['root_domain'].'%');
                            });
                        })
                        ;
                })->columnSpan(4)->columns(4)
            ])
            ->actions([
                Tables\Actions\Action::make("detail")->label(" ")
                    ->icon("heroicon-o-eye")
                    ->tooltip("View")
                    ->slideOver()
                    ->modalHeading(function ($record){
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return 'Completed';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return 'Blogger Rejected the Order';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return 'Waiting From Manager';
                        if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return 'Pending Order';
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return 'Rejected by Manager';
                    })

                    ->form([
                        Forms\Components\Placeholder::make("id")->label(" ")->content(function ($record){
                            $b = Carbon::parse($record->created_at)->addHours($record->new_site->total_time);
                            if($record->new_site->total_time){
                                if(Carbon::today()->gt($b)){
                                    return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Your project
                                    time is ticking away! Stay focused and make the most of the time left to complete
                                    your tasks. Don\'t let the clock run out on your progress!</span>');
                                }else{
                                    $timeDifference = Carbon::now()->diff($b);
                                    // Calculate the remaining days, hours, and minutes
                                    $remainingDays = $timeDifference->days;
                                    $remainingHours = $timeDifference->h;
                                    $remainingMinutes = $timeDifference->i;

                                    // Create an array to store the time components
                                    $timeComponents = [];

                                    if ($remainingDays > 0) {
                                        $timeComponents[] = $remainingDays . ($remainingDays === 1 ? ' Day' : ' Days');
                                    }
                                    if ($remainingHours > 0) {
                                        $timeComponents[] = $remainingHours . ($remainingHours === 1 ? ' Hour' : ' Hours');
                                    }
                                    if ($remainingMinutes > 0) {
                                        $timeComponents[] = $remainingMinutes . ($remainingMinutes === 1 ? ' Minute' : ' Minutes');
                                    }

                                    // Combine the time components into the desired format
                                    $remainingTime = implode(' : ', $timeComponents);

                                    return new HtmlString('Remaining Time:- '.$remainingTime);
                                }
                            }
                            return '';
                        }),

                        Forms\Components\Placeholder::make("status")->content(function ($record){
                            $detail = $record;
                            if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return new HtmlString('<span style="background: blue;color: white;padding:2px 8px;display: inline-block">Completed</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Blogger Rejected the Order</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Waiting From Manager</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Pending Order</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Rejected by Manager</span>');
                            return '';
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("order_id")->label("Order ID")->content(function ($record){
                            return $record->new_order_process->new_order->order_id;
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("root_domain")->label("Root Domain")->content(function ($record){
                            return $record->new_site->root_domain;
                        })->inlineLabel(),

                        //Order Type Niche
                        Forms\Components\Placeholder::make("urls")->label("Post Urls")->content(function ($record){
                            return $record->url;
                        })->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "niche"){
                                    return false;
                                }
                                return true;
                            })
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("insert_after")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace With' : 'Insert After';
                        })->content(function ($record){
                            return new HtmlString($record->insert_after);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),


                        Forms\Components\Placeholder::make("statement")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace Statement' : 'Insert Statement';
                        })->content(function ($record){
                            return new HtmlString($record->statement);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                        ->inlineLabel(),
                        //END Order Type Niche



                        //Order Type GP
                        Forms\Components\Placeholder::make("anchor")->label("Anchor")
                            ->content(fn ($record) => $record->anchor )
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("title")->label("Title")
                            ->content(fn ($record) => $record->title )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("doc_urls")->label("Doc Urls")
                            ->content(fn ($record) => new HtmlString('<span style="word-wrap: break-word">'.$record->doc_urls.'</span>') )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("upload_doc_file")->label("Doc File")
                            ->content(function($record){
                                $link = Str::startsWith($record->upload_doc_file,"uploads/") ? "/".$record->upload_doc_file : "/uploads/".$record->upload_doc_file;
//                                return strlen($record->upload_doc_file);
                                return new HtmlString('<a href="'.url($link) .'">View File</a>');
                            })
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp" && strlen($record->upload_doc_file) != 0) return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("ourl")->label("Url")
                            ->content(fn ($record) => $record->ourl )
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("note")->label("Note")
                            ->content(fn ($record) => $record->note )
                            ->inlineLabel(),

                        Toggle::make("upfront_payment")->inlineLabel()->disabled()->formatStateUsing(function ($record){
                            return $record->upfront_payment;
                        }),
                        Textarea::make("upfront_payment_remarks")->inlineLabel()
                            ->formatStateUsing(function ($record){
                                return $record->upfront_payment_remarks;
                            })
                            ->disabled()->hidden(function (Get $get){
                            if($get('upfront_payment')){
                                return false;
                            }
                            return true;
                        }),
                        TextInput::make("payment_method")
                            ->disabled()
                            ->hidden(function (Get $get){
                                if( $get('upfront_payment') ){
                                   return false;
                                }
                                return true;
                            })
                            ->inlineLabel()->formatStateUsing(function ($record){
                            return $record->payment_method;
                        }),
                        Forms\Components\Section::make("Bank Details")->schema([
                            TextInput::make("bank_type")
                                ->formatStateUsing(function ($record){
                                    return $record->bank_type;
                                })
                                ->label("Account Type")->disabled(),
                            TextInput::make("beneficiary_account_number")->label("Beneficiary Account Number")
                                ->formatStateUsing(function ($record){
                                    return $record->beneficiary_account_number;
                                })
                                ->disabled(),
                            TextInput::make("beneficiary_name")->label("Beneficiary Name")
                                 ->formatStateUsing(function ($record){
                                    return $record->beneficiary_name;
                                })
                                ->disabled(),
                            TextInput::make("bene_bank_name")->label("Bank Name")
                                 ->formatStateUsing(function ($record){
                                    return $record->bene_bank_name;
                                })
                                ->disabled(),

                            TextInput::make("ifsc_code")->label("IFSC Code")
                                 ->formatStateUsing(function ($record){
                                    return $record->ifsc_code;
                                })
                                ->disabled(),

                            TextInput::make("bene_bank_branch_name")->label("Bank Branch Name")
                                 ->formatStateUsing(function ($record){
                                    return $record->bene_bank_branch_name;
                                })
                                ->disabled(),
                            TextInput::make("beneficiary_email_id")->label("Beneficiary Email id")
                                 ->formatStateUsing(function ($record){
                                    return $record->beneficiary_email_id;
                                })
                                ->disabled(),
                            TextInput::make("customer_reference_number")->label("Customer Reference Number")
                                 ->formatStateUsing(function ($record){
                                    return $record->customer_reference_number;
                                })
                                ->disabled(),
                        ])
                            ->hidden(function (Get $get){
                                if( $get('upfront_payment') ){
                                    if($get('payment_method') == 'bank'){
                                        return false;
                                    }
                                    return true;
                                }
                                return true;
                            })
                            ->columns(3),

                        Forms\Components\Section::make("Paypal Details")->schema([
                            TextInput::make("paypal_id")
                                ->formatStateUsing(function ($record){
                                    return $record->paypal_email;
                                })
                                ->inlineLabel()->label("Paypal ID")

                                ->disabled()->required(),
                        ])
                            ->hidden(function (Get $get){
                                if( $get('upfront_payment') ){
                                    if($get('payment_method') == 'paypal_id'){
                                        return false;
                                    }
                                    return true;
                                }
                                return true;
                            }),


                        Forms\Components\Section::make("QR Code")->schema([
                            FileUpload::make("qr_code_image")
                                ->image()
                                ->required()
                                ->disabled()
                                ->default(function ($record){
                                    return $record->qr_code_image;
                                })
                                ->label("Upload QR Code Image"),
                        ])
                            ->hidden(function (Get $get){
                                if( $get('upfront_payment') ){
                                    if($get('payment_method') == 'qr_code'){
                                        return false;
                                    }
                                    return true;
                                }
                                return true;
                            }),

                        Forms\Components\Section::make("UPI Details")->schema([
                            TextInput::make("upi_id")
                                ->disabled()
                                ->formatStateUsing(function ($record){
                                    return $record->upi_id;
                                })
                                ->label("UPI Id")->required(),
                        ]) ->hidden(function (Get $get){
                            if( $get('upfront_payment') ){
                                if($get('payment_method') == 'upi_id'){
                                    return false;
                                }
                                return true;
                            }
                            return true;
                        }),


                        Forms\Components\Placeholder::make("submit_url")->label("Submit URL")
                            ->content(fn ($record) => $record->submit_url )
                            ->hidden(function ($record){
                                $detail = $record;
                                if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return false;

                                return true;
                            })
                            ->inlineLabel(),

                        TextInput::make("submit_url")->label("Submit URL")->required()->url()


                    ])
                    ->action(function ($record, $data){



                        try{
                            DB::beginTransaction();
                            $detail = $record;
                            $v =( new NewLinkVerify(trim($detail->ourl), trim($detail->anchor), trim($data['submit_url'])))->verify();


                            $detail->submit_url= $data['submit_url'];
                            $detail->verify = $v ? "verified" : "Not Verified";
                            $detail->status = NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS;
                            $detail->save();


                            Notification::make("success")->title("Successfully Pushed to Manager")->success()->send();

                            $manager= User::where('role','manager')->where('id',$record->new_order_process->new_order->manager_id)->first();
                            $admin = User::where('role','admin')->first();

                            Notification::make()
                                ->title("Order Pushed by ".Auth::user()->name.'('.Auth::user()->email.')')
                                ->body("Order No: ".$record->new_order_process->new_order->order_id)
                                ->success()
                                ->sendToDatabase([$manager,$admin]);


                            DB::commit();
                            redirect()->to(OrderResource::getUrl());
                        }catch (\Exception $exception){
                            DB::rollBack();
                            dd($exception->getMessage());
                        }
                    })
                    ->hidden(function ($record) {
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER || $detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER){
                            return false;
                        }
                        return true;
                    }),

                Tables\Actions\Action::make("rejection")->label(" ")
                    ->icon("heroicon-o-trash")
                    ->color('danger')
                    ->tooltip("Reject Order")
                    ->slideOver()
                    ->modalHeading(function ($record){
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return 'Completed';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return 'Blogger Rejected the Order';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return 'Waiting From Manager';
                        if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return 'Pending Order';
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return 'Rejected by Manager';
                    })

                    ->form([

                        Forms\Components\Placeholder::make("status")->content(function ($record){
                            $detail = $record;
                            if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return new HtmlString('<span style="background: blue;color: white;padding:2px 8px;display: inline-block">Completed</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Blogger Rejected the Order</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Waiting From Manager</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Pending Order</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Rejected by Manager</span>');
                            return '';
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("order_id")->label("Order ID")->content(function ($record){
                            return $record->new_order_process->new_order->order_id;
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("root_domain")->label("Root Domain")->content(function ($record){
                            return $record->new_site->root_domain;
                        })->inlineLabel(),

                        //Order Type Niche
                        Forms\Components\Placeholder::make("urls")->label("Post Urls")->content(function ($record){
                            return $record->url;
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("insert_after")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace With' : 'Insert After';
                        })->content(function ($record){
                            return new HtmlString($record->insert_after);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),


                        Forms\Components\Placeholder::make("statement")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace Statement' : 'Insert Statement';
                        })->content(function ($record){
                            return new HtmlString($record->statement);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),
                        //END Order Type Niche



                        //Order Type GP
                        Forms\Components\Placeholder::make("anchor")->label("Anchor")
                            ->content(fn ($record) => $record->anchor )
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("title")->label("Title")
                            ->content(fn ($record) => $record->title )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("doc_urls")->label("Doc Urls")
                            ->content(fn ($record) => new HtmlString('<span style="word-wrap: break-word">'.$record->doc_urls.'</span>') )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("upload_doc_file")->label("Doc File")


                            ->content(function ($record){
                                $link = Str::startsWith($record->upload_doc_file,"uploads/") ? "/".$record->upload_doc_file : "/uploads/".$record->upload_doc_file;
//                                return strlen($record->upload_doc_file);
                                return new HtmlString('<a href="'.url($link) .'">View File</a>');
                            })
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp" && strlen($record->upload_doc_file) != 0) return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("ourl")->label("Url")
                            ->content(fn ($record) => $record->ourl )
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("note")->label("Note")
                            ->content(fn ($record) => $record->note )
                            ->inlineLabel(),

                        Toggle::make("upfront_payment")->inlineLabel()->disabled()->formatStateUsing(function ($record){
                            return $record->upfront_payment;
                        }),
                        Textarea::make("upfront_payment_remarks")->inlineLabel()
                            ->formatStateUsing(function ($record){
                                return $record->upfront_payment_remarks;
                            })
                            ->disabled()->hidden(function (Get $get){
                                if($get('upfront_payment')){
                                    return false;
                                }
                                return true;
                            }),


                        Forms\Components\Placeholder::make("submit_url")->label("Submit URL")
                            ->content(fn ($record) => $record->submit_url )
                            ->hidden(function ($record){
                                $detail = $record;
                                if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return false;

                                return true;
                            })
                            ->inlineLabel(),

                        TextInput::make("reason_for_rejection")->label("Reason for Rejection")->required()


                    ])
                    ->action(function ($record, $data){
                        try{
                            $detail = $record;
                            $detail->status = NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION;
                            $detail->reject_reason = $data['reason_for_rejection'];
                            $detail->save();

                            Notification::make("success")->title("Order Rejected")->success()->send();

                            $manager= User::where('role','manager')->where('id',$record->new_order_process->new_order->manager_id)->first();
                            $admin = User::where('role','admin')->first();

                            Notification::make()
                                ->title("Order Rejected by ".Auth::user()->name.'('.Auth::user()->email.')')
                                ->body("Order No: ".$record->new_order_process->new_order->order_id)
                                ->success()
                                ->sendToDatabase([$manager,$admin]);


                            redirect()->to(OrderResource::getUrl());
                        }catch (\Exception $exception){
                            dd($exception->getMessage());
                        }
                    })
                    ->hidden(function ($record) {
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER || $detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER){
                            return false;
                        }
                        return true;
                    }),

                Tables\Actions\Action::make("info")->label(" ")
                    ->icon("heroicon-o-eye")
                    ->tooltip("Detail")
                    ->slideOver()
                    ->modalHeading(function ($record){
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return 'Completed';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return 'Blogger Rejected the Order';
                        if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return 'Waiting From Manager';
                        if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return 'Pending Order';
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return 'Rejected by Manager';
                    })
                    ->modalSubmitAction(false)
                    ->modalCancelAction(false)
                    ->form([



                        Forms\Components\Placeholder::make("status")->content(function ($record){
                            $detail = $record;
                            if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return new HtmlString('<span style="background: blue;color: white;padding:2px 8px;display: inline-block">Completed</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Blogger Rejected the Order</span>');
                            if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Waiting From Manager</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER) return new HtmlString('<span style="background: orange;color: white;padding:2px 8px;display: inline-block">Pending Order</span>');
                            if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER) return new HtmlString('<span style="background: red;color: white;padding:2px 8px;display: inline-block">Rejected by Manager</span>');
                            return '';
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("order_id")->label("Order ID")->content(function ($record){
                            return $record->new_order_process->new_order->order_id;
                        })->inlineLabel(),

                        Forms\Components\Placeholder::make("root_domain")->label("Root Domain")->content(function ($record){
                            return $record->new_site->root_domain;
                        })->inlineLabel(),

                        //Order Type Niche
                        Forms\Components\Placeholder::make("urls")->label("Post Urls")->content(function ($record){
                            return $record->url;
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("insert_after")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace With' : 'Insert After';
                        })->content(function ($record){
                            return new HtmlString($record->insert_after);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),


                        Forms\Components\Placeholder::make("statement")->label(function ($record){
                            $detail = $record;
                            return ($detail->type == 'replace') ? 'Replace Statement' : 'Insert Statement';
                        })->content(function ($record){
                            return new HtmlString($record->statement);
                        })->hidden(function ($record){
                            if($record->new_order_process->new_order->order_type == "niche"){
                                return false;
                            }
                            return true;
                        })
                            ->inlineLabel(),
                        //END Order Type Niche



                        //Order Type GP
                        Forms\Components\Placeholder::make("anchor")->label("Anchor")
                            ->content(fn ($record) => $record->anchor )
                            ->inlineLabel(),

                        Forms\Components\Placeholder::make("title")->label("Title")
                            ->content(fn ($record) => $record->title )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("doc_urls")->label("Doc Urls")
                            ->content(fn ($record) => new HtmlString('<span style="word-wrap: break-word">'.$record->doc_urls.'</span>') )
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp") return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("upload_doc_file")->label("Doc File")
                            ->content(function ($record){
                                $link = Str::startsWith($record->upload_doc_file,"uploads/") ? "/".$record->upload_doc_file : "/uploads/".$record->upload_doc_file;
//                                return strlen($record->upload_doc_file);
                                return new HtmlString('<a href="'.url($link) .'">View File</a>');
                            })
                            ->hidden(function ($record){
                                if($record->new_order_process->new_order->order_type == "gp" && strlen($record->upload_doc_file) != 0) return false;
                                return true;
                            })->inlineLabel(),

                        Forms\Components\Placeholder::make("ourl")->label("Url")
                            ->content(fn ($record) => $record->ourl )
                            ->inlineLabel(),


                        Forms\Components\Placeholder::make("note")->label("Note")
                            ->content(fn ($record) => $record->note )
                            ->inlineLabel(),

                        Toggle::make("upfront_payment")->inlineLabel()->disabled()->formatStateUsing(function ($record){
                            return $record->upfront_payment;
                        }),
                        Textarea::make("upfront_payment_remarks")->inlineLabel()
                            ->formatStateUsing(function ($record){
                                return $record->upfront_payment_remarks;
                            })
                            ->disabled()->hidden(function (Get $get){
                                if($get('upfront_payment')){
                                    return false;
                                }
                                return true;
                            }),


                        Forms\Components\Placeholder::make("submit_url")->label("Submit URL")
                            ->content(fn ($record) => $record->submit_url )
                            ->hidden(function ($record){
                                $detail = $record;
                                if($detail->status == NewOrderStatus::$MANAGER_VERIFIED_AND_FINISHED) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION) return false;
                                if($detail->status == NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS) return false;

                                return true;
                            })
                            ->inlineLabel(),



                    ])
                    ->action(function ($record, $data){

                    })
                    ->hidden(function ($record) {
                        $detail = $record;
                        if($detail->status == NewOrderStatus::$MANAGER_DISAPPROVED_AND_PUSHED_BACK_TO_BLOGGER || $detail->status == NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER){
                            return true;
                        }
                        return false;
                    })

            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrders::route('/'),
            'create' => Pages\CreateOrder::route('/create'),
            'edit' => Pages\EditOrder::route('/{record}/edit'),
        ];
    }
}
