<?php

namespace App\Filament\Writer\Resources;

use App\Filament\Writer\Resources\TicketResource\Pages;
use App\Filament\Writer\Resources\TicketResource\RelationManagers;
use App\Models\Thread;
use App\Models\Ticket;
use App\Models\User;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;

class TicketResource extends Resource
{
    protected static ?string $model = Thread::class;

    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-left-right';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make("Add New Ticket")
                    ->schema([
                        Forms\Components\Select::make("role")->options([
                            "manager"    =>  "Manager",
                        ])->live()
                            ->required()->inlineLabel(),
                        Forms\Components\Select::make("user")->options(function (Forms\Get $get){
                            if($get('role') == 'manager'){
                                return User::where('role','manager')->get()->pluck('name','id');
                            }
                            return [];
                        })->required()->inlineLabel(),
                        Forms\Components\TextInput::make("subject")->required()->inlineLabel()
                    ])
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                Thread::where('owner_id', Auth::id())->orWhere('user_id',Auth::id())->with("last_updated")->withCount("messages")
            )
            ->recordUrl(function (Thread $record) {
                return TicketResource::getUrl('details', ['record' => $record->id]);
            })
            ->columns([
                Tables\Columns\TextColumn::make("owner.name")->toggleable()->label("Thread Created by")->formatStateUsing(function (Thread $thread) {
                    return $thread->owner->name.' ('.$thread->owner->role.')';
                })->description(function (Thread $record) {
                    return 'Messages:- '.$record->messages_count;
                }),
                Tables\Columns\TextColumn::make("subject")->toggleable()->wrap(),
                Tables\Columns\TextColumn::make("id")->toggleable()->label("Last Updated at")->formatStateUsing(function (Thread $thread) {
                    if($thread->last_updated){
                        return Carbon::parse($thread->last_updated->created_at)->diffForHumans();
                    }else{
                        return Carbon::parse($thread->created_at)->diffForHumans();
                    }
                }),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\Action::make("detail")->url(function (Thread $record) {
                    return TicketResource::getUrl('details', ['record' => $record->id]);
                })
            ])
            ->bulkActions([
//                Tables\Actions\BulkActionGroup::make([
//                    Tables\Actions\DeleteBulkAction::make(),
//                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTickets::route('/'),
            'create' => Pages\CreateTicket::route('/create'),
            'edit' => Pages\EditTicket::route('/{record}/edit'),
            'details'   =>  Pages\ThreadDetails::route("/{record}/details"),
        ];
    }
}


