<?php

namespace App\Filament\Writer\Resources;

use App\Filament\Writer\Resources\OrderAddedNotificationResource\Pages;
use App\Filament\Writer\Resources\OrderAddedNotificationResource\RelationManagers;
use App\Models\NewOrder;
use App\Models\OrderAddedNotification;
use App\Models\User;
use App\Project\NewOrderStatus;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;

class OrderAddedNotificationResource extends Resource
{
    protected static ?string $model = NewOrder::class;

    protected static ?string $label = "Order Added Notification";

    protected static ?string $navigationIcon = 'heroicon-o-book-open';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function getNavigationBadge(): ?string
    {
        return writer_order_added_notifications();
    }


    public static function table(Table $table): Table
    {
        return $table
            ->query(
                NewOrder::whereHas("order_processes_manager_pushed_to_writer", function ($q){
                    $q->where('writer_id',Auth::id());
                })->has("order_processes_manager_pushed_to_writer.new_order_process_details")
                    ->where('new_order_status',NewOrderStatus::$MANAGER_PUSHED_TO_WRITER)
                    ->with("order_processes_manager_pushed_to_writer","manager")
            )
            ->recordUrl(null)
            ->columns([
                Tables\Columns\TextColumn::make("order_id")->label("Order Id"),
                Tables\Columns\TextColumn::make("order_type")->toggleable()->label("Order Type")->sortable(),
                Tables\Columns\TextColumn::make("manager.name")->toggleable()->sortable(),
                Tables\Columns\TextColumn::make("id")->toggleable()->label("Pushed Date")->formatStateUsing(function (NewOrder $order) {
                    return Carbon::parse($order->order_processes_manager_pushed_to_writer->created_at)->toDayDateTimeString();
                }),

            ])
            ->filtersLayout(FiltersLayout::AboveContentCollapsible)
            ->filters([
                DateRangeFilter::make("id")->label("Pushed Date")

                    ->modifyQueryUsing(function (Builder $query, ?Carbon $startDate, ?Carbon $endDate,$dateString) {

                        $query->when(!empty($dateString), function (Builder $query, $date) use ($startDate, $endDate) {
                            $query->whereHas('order_processes_manager_pushed_to_writer', function ($q) use ($startDate, $endDate) {
                                $q->whereBetween('created_at',[$startDate." 00:00:00",$endDate." 23:59:59"]);
                            });
                        });
                    }),
                Tables\Filters\Filter::make("Search")->form([
                    Forms\Components\TextInput::make("order_id"),
                    Forms\Components\Select::make("order_type")->options([
                        "gp"    =>  "GP",
                        "niche" =>  "Niche",
                    ]),
                    Forms\Components\Select::make("manager")->options(User::where('role','manager')->pluck("email","id"))


                ])

                    ->query(function (Builder $query, array $data): Builder {
                        return $query

                            ->when($data['order_id'],function (Builder $query) use ($data){
                                return $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            })
                            ->when($data['order_type'],function (Builder $query) use ($data){
                                return $query->where('order_type',$data['order_type']);
                            })
                            ->when($data['manager'],function (Builder $query) use ($data){
                                return $query->where('manager_id',$data['manager']);
                            })
                            ;
                    })
                    ->columns(6)->columnSpanFull()


            ])
            ->actions([
                Tables\Actions\Action::make('Detail')
                    ->icon("heroicon-o-arrow-top-right-on-square")->label(" ")->tooltip("Push To Manager")
                    ->url(function (NewOrder $record) {
                    return OrderAddedNotificationResource::getUrl('detail', ['record' => $record->id]);
                })
            ])
            ->bulkActions([

            ]);
    }



    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrderAddedNotifications::route('/'),
            'create' => Pages\CreateOrderAddedNotification::route('/create'),
            'edit' => Pages\EditOrderAddedNotification::route('/{record}/edit'),
            'detail' => Pages\OrderAddedNotificationDetailPage::route("/detail/{record}"),
        ];
    }
}
