<?php

namespace App\Filament\Resources;

use App\Filament\Pages\UsersPermissions;
use App\Filament\Resources\TeamMemberResource\Pages;
use App\Filament\Resources\TeamMemberResource\RelationManagers;
use App\Models\TeamMember;
use App\Models\User;
use Carbon\Carbon;
use Filament\Facades\Filament;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;

class TeamMemberResource extends Resource
{
    protected static ?string $model = User::class;
    protected static ?string $navigationLabel = "Team Members";
    protected static ?string $navigationIcon = 'heroicon-o-users';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
               Forms\Components\Section::make("Team Member")->schema([

                   Forms\Components\Select::make("role")
                       ->options([
                           "team" => "Team",
                           "writer"    =>  "Writer",
                           "manager"   =>  "Manager",
                       ])->required(),

                   Forms\Components\TextInput::make("name")->required(),
                   Forms\Components\TextInput::make("email")->email()->unique()->required(),
                   Forms\Components\TextInput::make("password")->password()->required(),

               ])->columns(4)
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                User::whereIn('role',['team','manager','writer'])
            )
            ->paginated(config('app.pagination'))
            ->columns([
                Tables\Columns\TextColumn::make("name")->searchable()->sortable(),
                Tables\Columns\TextColumn::make("email")->searchable()->sortable(),

                Tables\Columns\TextColumn::make("role")->sortable(),

                Tables\Columns\TextColumn::make("current_login")->label("Last Login")->formatStateUsing(function (User $record) {
                    return $record->current_login ? Carbon::parse($record->current_login)->toDayDateTimeString() : "";
                })->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make("login_count")->label("Login Counts")->formatStateUsing(function (User $record) {
                    return $record->login_count ?? 0;
                })->sortable(),
                Tables\Columns\TextColumn::make("status")->label("Account Status")->formatStateUsing(function (User $record) {
                    return $record->status ? "Active" : "Blocked";
                })->badge()
                    ->color(function (User $record) {
                        if($record->status == 1){
                            return "success";
                        }else{
                            return "danger";
                        }
                    })
                    ->action(function (User $record) {
                    if( $record->status == 1 ){
                        $record->status = 0;
                    }else{
                        $record->status = 1;
                    }
                    $record->save();
                })
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                Tables\Filters\Filter::make('role')
                    ->form([
                        Forms\Components\Select::make("role")
                            ->options([
                                "team" => "Team",
                                "writer"    =>  "Writer",
                                "manager"   =>  "Manager",
                            ]),
                        Forms\Components\TextInput::make("name"),
                        Forms\Components\TextInput::make("email"),
                    ])
                    ->query(function (Builder $query, array $data) {
                        return $query->when($data['role'],function ($q) use ($data){
                            return $q->where('role', $data['role']);
                        })->when($data['name'], function (Builder $query) use ($data) {
                            return $query->where('name', 'LIKE','%'.$data['name'].'%');
                        })
                            ->when($data['email'], function (Builder $query) use ($data) {
                                return $query->where('email', 'LIKE','%'.$data['email'].'%');
                            })

                            ;
                    })->columns(6)->columnSpanFull()
            ])
            ->actions([
                Tables\Actions\Action::make("reset_password")
                    ->icon("heroicon-o-arrow-path-rounded-square")
                    ->label(" ")
                    ->tooltip("Reset Password to 12345678")
                    ->action(function (User $record) {
                        $record->password = bcrypt("12345678");
                        $record->save();
                        Notification::make("success")->title("Password Reset")->send();
                    })
                ,
                Tables\Actions\Action::make("permissions")->url(function (User $record) {
                    return UsersPermissions::getUrl(["id" => $record->id]);
                })->icon("heroicon-s-arrow-left-end-on-rectangle")->label(" ")->tooltip("View Permissions"),

                Tables\Actions\Action::make("change_password")->label(" ")->tooltip("Change Password")->icon("heroicon-s-cog-8-tooth")->form([
                    Forms\Components\TextInput::make("password")
                ])->action(function (User $record, array $data) {
                    try{
                        $record->password = bcrypt($data['password']);
                        $record->save();
                        Notification::make("success")->title("Password Changed")->success()->body("Your Password has been changed successfully")->send();
                    }catch (\Exception $exception){
                        dd($exception->getMessage());
                    }
                })

            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTeamMembers::route('/'),
            'create' => Pages\CreateTeamMember::route('/create'),
            'edit' => Pages\EditTeamMember::route('/{record}/edit'),
        ];
    }
}
