<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OrderResource\Pages;
use App\Filament\Resources\OrderResource\RelationManagers;
use App\Models\NewOrder;
use App\Models\Order;
use App\Models\OrdersView;
use App\Project\NewOrderStatus;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class OrderResource extends Resource
{
    protected static ?string $model = OrdersView::class;

    protected static ?string $navigationLabel = "Orders";

    protected static ?string $navigationIcon = 'heroicon-o-credit-card';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                OrdersView::with("new_order_processes_many")
            )
            ->recordUrl(null)
            ->columns([
                Tables\Columns\TextColumn::make("order_id")->sortable()->description(function (OrdersView $order){
                    if($order->type == 'new'){
                        return 'New Order';
                    }
                    if($order->type == 'sub'){
                        return 'Sub Order';
                    }
                }),
                Tables\Columns\TextColumn::make("name")->toggleable()->label("Manager")->sortable()->wrap(),
                Tables\Columns\TextColumn::make("client_name")->toggleable()->label("Client Name")->wrap()->sortable(),
                Tables\Columns\TextColumn::make("total")->toggleable()->label("Status")
                    ->badge()
                    ->color(function (OrdersView $order){
                        if(($order->total == $order->total1) && $order->total != 0 && $order->total1 !=0 ){
                            return 'success';
                        }elseif( ( $order->new_order_status == NewOrderStatus::$PENDING_ORDER ) || ($order->new_order_status == NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM) ){
                            return 'warning';
                        }else{
                            return 'danger';
                        }
                    })
                    ->formatStateUsing(function (OrdersView $order) {
                        $text = '';
                        if(($order->total == $order->total1) && $order->total != 0 && $order->total1 !=0 ){
                            $text  .=' Completed';
                        }elseif( ( $order->new_order_status == NewOrderStatus::$PENDING_ORDER ) || ($order->new_order_status == NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM) ){
                            $text  .='Pending';
                        }else{
                            $text  .='In Process';
                        }
                        return new HtmlString($text);
                    })->description(function (OrdersView $order){
                        return new_order_status($order->new_order_status);
                    }),
                Tables\Columns\TextColumn::make("client_website")
                    ->toggleable()
                    ->icon("heroicon-o-document-duplicate")
                    ->tooltip("Click here to Copy")
                    ->label("Website")->copyable()->wrap(),
                Tables\Columns\TextColumn::make("id")->label("TAT")->formatStateUsing(function (OrdersView $order) {
                    if(($order->total == $order->total1) && $order->total != 0 && $order->total1 !=0 ){
                            return '';
                    }else{
                        $count = 0;
                        foreach ($order->new_order_processes_many as $item){
                            $b = Carbon::parse($item->created_at)->addHours($item->new_site->total_time);
                            if(Carbon::today()->gt($b)){
                                $count += 1;
                            }
                        }
                        return $count;
                    }
                }),
                Tables\Columns\TextColumn::make("no_of_links")->toggleable()->label("No of Links")->sortable(),
                Tables\Columns\TextColumn::make("order_type")->toggleable()->label("Order Type")->sortable(),
                Tables\Columns\TextColumn::make("order_package")->toggleable()->wrap()->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make("category")->toggleable()->wrap()->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make("created_at")->toggleable()->sortable()->label("Ordered At")->formatStateUsing(function (OrdersView $order) {
                    return Carbon::parse($order->created_at)->toDateTimeString();
                }),
                Tables\Columns\TextColumn::make("type")->label("Completed At")->formatStateUsing(function (OrdersView $order) {
                    if(($order->total == $order->total1) && $order->total != 0 && $order->total1 !=0 ){
                        $d = collect($order->new_order_processes_many)
                            ->sortByDesc('updated_at')
                            ->values()
                            ->first();
                        if($d){
                            return $d->updated_at;
                        }

                        return "";
                    }
                }),

            ])
            ->paginated(config('app.pagination'))
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForFilename()->askForWriterType()->withColumns([
                        Column::make("order_id"),
                        Column::make("name")->heading("Manager"),
                        Column::make("total")->heading("Status")->formatStateUsing(function ( $record) {
                            if(($record->total == $record->total1) && $record->total != 0 && $record->total1 !=0 ){
                                return  'Completed';
                            }elseif( ( $record->new_order_status == NewOrderStatus::$PENDING_ORDER ) || ($record->new_order_status == NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM) ){
                                return 'Pending';
                            }else{
                                return 'In Process';
                            }
                        }),
                        Column::make("total1")->heading("Status Description")->formatStateUsing(function ($record){
                            return new_order_status($record->new_order_status);
                        }),
                        Column::make("client_name"),
                        Column::make("client_website"),
                        Column::make("no_of_links"),
                        Column::make("order_type"),
                        Column::make("order_package"),
                        Column::make("category"),
                        Column::make("created_at")->heading("Ordered At"),

                    ])
                ])
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                DateRangeFilter::make("created_at")->label("Ordered At"),
                Tables\Filters\Filter::make("Search")->form([
                    Forms\Components\TextInput::make("order_id"),
                    Forms\Components\TextInput::make("category"),
                    Forms\Components\TextInput::make("client_name"),
                    Forms\Components\TextInput::make("client_website"),
                    Forms\Components\Select::make("status")->options([
                        "completed" => "Completed",
                        "in_process" => "In Process",
                        "pending" => "Pending",
                        "manager" => "Manager",
                        "team" => "Team",
                        "writer" => "Writer",
                        "blogger" => "Blogger",

                    ]),

                ])

                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['status'],function (Builder $query) use ($data){
                                if($data['status'] == 'completed'){
                                    return $query->whereRaw('total = total1 and total > 0');
                                }
                                if( $data['status'] == 'in_process' ){
                                    return $query->whereRaw("new_order_status != 0 and new_order_status != 21 and ( (total != total1 and total > 0) or (total = total1 and total = 0) )");

                                }
                                if( $data['status'] == 'pending' ){
                                    return $query->whereIn('new_order_status',[NewOrderStatus::$PENDING_ORDER,NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM]);
                                }

                                if( $data['status'] == 'team' ){
                                    return $query->whereIn('new_order_status',[NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM,NewOrderStatus::$PENDING_ORDER,NewOrderStatus::$MANAGER_DISAPPROVED_POST_TEAM]);
                                }

                                if( $data['status'] == 'writer' ){
                                    return $query->whereIn('new_order_status',[NewOrderStatus::$MANAGER_PUSHED_TO_WRITER,NewOrderStatus::$MANAGER_REJECTED_AND_PUSHED_BACK_TO_WRITER]);
                                }

                                if( $data['status'] == 'manager' ){
                                    return $query->whereIn('new_order_status',[NewOrderStatus::$TEAM_PUSHED_TO_MANAGER,NewOrderStatus::$WRITER_PUSHED_TO_MANAGER]);
                                }



                                if( $data['status'] == 'blogger' ){
                                    return $query->whereIn('new_order_status',[NewOrderStatus::$MANAGER_PUSHED_TO_BLOGGER]);
                                }


                            })
                            ->when('order_id',function (Builder $query) use ($data){
                                return $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            })
                            ->when('category',function (Builder $query) use ($data){
                                return $query->where('category','LIKE','%'.$data['category'].'%');
                            })
                            ->when('client_name',function (Builder $query) use ($data){
                                return $query->where('client_name','LIKE','%'.$data['client_name'].'%');
                            })
                            ->when('client_website',function (Builder $query) use ($data){
                                return $query->where('client_website','LIKE','%'.$data['client_website'].'%');
                            })
                            ;
                    })
                    ->columns(6)->columnSpanFull()
            ])
            ->actions([
                Tables\Actions\Action::make("detail")
                    ->label(" ")
                    ->icon("heroicon-s-eye")
                    ->tooltip("Detail")
                    ->url(function (OrdersView $record) {
                        return OrderResource::getUrl('detail', ['record' => $record->id]);
                    })
            ])
          ;
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrders::route('/'),
            'create' => Pages\CreateOrder::route('/create'),
            'edit' => Pages\EditOrder::route('/{record}/edit'),
            "detail"    =>  Pages\OrderDetail::route('/{record}'),
        ];
    }
}
