<?php

namespace App\Filament\Pages;

use App\Models\NewSite;
use App\Models\WithdrawRequest;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Infolists\Components\TextEntry;
use Filament\Pages\Page;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\Summarizers\Summarizer;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class PaymentHistory extends Page implements HasForms, HasTable
{
    use InteractsWithTable;
    use InteractsWithForms;
    protected static ?string $navigationGroup = "Wallet";
    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';

    protected static string $view = 'filament.pages.payment-history';


    public function table(Table $table): Table
    {
        return $table
            ->query(
                WithdrawRequest::withSum(['wallet_histories as total_price' => function ($query) {
                    $query->select(DB::raw('SUM(price)'));
                }], 'price')->where('status',1)
            )
            ->defaultSort("updated_at","desc")
            ->paginated(config('app.pagination'))
            ->columns([

                TextColumn::make('user.name')->description(function (WithdrawRequest $record) {
                    return $record->user->email;
                })->wrap(),
                TextColumn::make("created_at")
                    ->toggleable()
                    ->label("Payment Method")
                    ->wrap()
                    ->description(function (WithdrawRequest $record) {
                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return new HtmlString('
                                <strong>Bank Type:- </strong>'.$record->wallet_histories[0]->bank_type.' <br>
                                <strong>Beneficiary Account Number:- </strong>'.$record->wallet_histories[0]->beneficiary_account_number.' <br>
                                <strong>Beneficiary Name:- </strong>'.$record->wallet_histories[0]->beneficiary_name.' <br>
                                <strong>Customer Reference Number:- </strong>'.$record->wallet_histories[0]->customer_reference_number.' <br>
                                <strong>Beneficiary Bank Name:- </strong>'.$record->wallet_histories[0]->bene_bank_name.' <br>
                                <strong>Beneficiary Bank Branch Name:- </strong>'.$record->wallet_histories[0]->bene_bank_branch_name.' <br>
                                <strong>IFSC Code:- </strong>'.$record->wallet_histories[0]->ifsc_code.' <br>
                                <strong>Beneficiary Email id:- </strong>'.$record->wallet_histories[0]->beneficiary_email_id .' <br>'

                            );
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
//                            $qr_code_image = Str::startsWith($record->wallet_histories[0]->qr_code_image,"uploads/") ? "/".$record->wallet_histories[0]->qr_code_image : "/uploads/".$record->wallet_histories[0]->qr_code_image;
                            $qr_code_image = "/".$record->wallet_histories[0]->qr_code_image;
                            return new HtmlString('<div class="flex items-center flex-col justify-between border-b border-gray-200 mt-2 pb-1 mb-1" >

                                    <span class="text-black font-bold" >
                                    <a href = "'.$qr_code_image .'" target = "_blank" >
                                        <img src = "'.$qr_code_image .'" class="w-32" >
                                    </a >
                                    </span >
                                </div >');
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return $record->wallet_histories[0]->paypal_email;

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return $record->wallet_histories[0]->upi_id;

                        }

                    })
                    ->badge()
                    ->color(function (WithdrawRequest $record) {
                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return 'success';
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
                            return 'danger';
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return 'info';

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return 'warning';

                        }

                    })
                    ->formatStateUsing(function (WithdrawRequest $record) {

                        if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                            return 'Bank Details';
                        }


                        if($record->wallet_histories[0]->payment_method == "qr_code") {
                            return 'QR Code';
                        }

                        if($record->wallet_histories[0]->payment_method == "paypal_id") {
                            return 'Paypal ID';

                        }

                        if($record->wallet_histories[0]->payment_method == "upi_id") {
                            return 'UPI ID';

                        }



                }),
                TextColumn::make("total_price")
                    ->toggleable()
                    ->label("Amount")->summarize(Sum::make())->sortable(),

                TextColumn::make("updated_at")->toggleable()->label("Datetime")->wrap()->formatStateUsing(function (WithdrawRequest $record) {
                    return new HtmlString(' <div class="flex items-center">
                                <div style="font-weight: bold;font-size: 12px;">Request Date :- </div>
                                <div style="font-size: 12px;">'.Carbon::parse($record->wallet_histories[0]->request_date)->toDayDateTimeString() .'</div>
                            </div>
                             <div class="flex items-center">
                                <div style="font-weight: bold;font-size: 12px;">Clearance Date :-</div>
                                <div style="font-size: 12px;">'.Carbon::parse($record->wallet_histories[0]->approved_date)->toDayDateTimeString().'</div>
                            </div>
                            <div class="flex items-center">
                                <div style="font-weight: bold;font-size: 12px;">Remarks:- </div>
                                <div style="font-size: 12px;">'.$record->wallet_histories[0]->remarks.'</div>
                            </div>


                            </div>
');
                }),

            ])
            ->filters([
                Filter::make("search")->form([
                    TextInput::make("name"),
                    TextInput::make("email"),
                    Select::make("payment_method")->options([
                        "bank" => "Bank",
                        "qr_code" => "QR Code",
                        "paypal_id" => "Paypal ID",
                        "upi_id" => "UPI ID",
                    ])
                ])->columns(6)->columnSpanFull()
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['name'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('name', 'LIKE','%'.$data['name'].'%');
                                });
                            })
                            ->when($data['email'], function (Builder $query) use ($data) {
                                return $query->whereHas("user", function (Builder $query) use ($data) {
                                    $query->where('email', 'LIKE','%'.$data['email'].'%');
                                });
                            })

                            ->when($data['payment_method'], function (Builder $query) use ($data) {
                                return $query->whereHas("wallet_histories", function (Builder $query) use ($data) {
                                    return $query->where('payment_method', $data['payment_method']);
                                });
                            });
                    }),
                DateRangeFilter::make("created_at")->label("Clearance Date")
                    ->modifyQueryUsing(function (Builder $query, ?Carbon $startDate, ?Carbon $endDate, $dateString) {
                        $query->when(!empty($dateString), function (Builder $query, $date) use ($startDate, $endDate) {
                            $query->whereHas("wallet_histories", function (Builder $query) use ($date,$startDate,$endDate){
                                return $query->whereBetween('approved_date', [$startDate,$endDate]);
                            });
                        });
                    })
                ,
            ])
            ->filtersLayout(FiltersLayout::AboveContentCollapsible)
            ->actions([
//                Action::make("detail")->url(function (WithdrawRequest $record) {
//                    return PaymentHistoryDetail::getUrl(["id" => $record->id]);
//                })

                Action::make("detail")
                    ->label(" ")
                    ->icon("heroicon-s-eye")
                    ->tooltip("Detail")
                    ->modalContent(function (WithdrawRequest $record) {
                    return view("welcome")->with("record", $record->id)->with("action", "update");
                })->modalSubmitAction(false)->modalCancelAction(false)

            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForWriterType()->askForFilename()->withColumns([
                        Column::make("user.name")->heading("Name"),
                        Column::make("user.email")->heading("Email"),
                        Column::make("created_at")->heading("Payment Method")->formatStateUsing(function ($record){
                            if( $record->wallet_histories[0]->payment_method  == 'bank' ){
                                return 'Bank Details';
                            }


                            if($record->wallet_histories[0]->payment_method == "qr_code") {
                                return 'QR Code';
                            }

                            if($record->wallet_histories[0]->payment_method == "paypal_id") {
                                return 'Paypal ID';

                            }

                            if($record->wallet_histories[0]->payment_method == "upi_id") {
                                return 'UPI ID';

                            }
                        }),

                        Column::make("user.id")->heading("Paypal Id")->formatStateUsing(function ($record){
                            return $record->wallet_histories[0]->paypal_email;
                        }),
                        Column::make("user.created_at")->heading("UPI Id")->formatStateUsing(function ($record){
                            return $record->wallet_histories[0]->upi_id;
                        }),

                        Column::make("user.updated_at")->heading("QR Code")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->qr_code_image; }),

                        Column::make("user.beneficiary_account_number")->heading("Bank Type")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->bank_type;  }),


                        Column::make("user.beneficiary_name")->heading("Beneficiary Account Number")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->beneficiary_account_number;  }),
                        Column::make("user.customer_reference_number")->heading("Beneficiary Name")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->beneficiary_name;  }),
                        Column::make("user.ifsc_code")->heading("Customer Reference Number")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->customer_reference_number;  }),
                        Column::make("user.bene_bank_name")->heading("Beneficiary Bank Name")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->bene_bank_name;  }),
                        Column::make("user.bene_bank_branch_name")->heading("Beneficiary Bank Branch Name")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->bene_bank_branch_name;  }),
                        Column::make("user.beneficiary_email_id")->heading("IFSC Code")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->ifsc_code;  }),
                        Column::make("user.bank_type")->heading("Beneficiary Email id")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->beneficiary_email_id; }),
                        Column::make("id")->heading("Amount")->formatStateUsing(function ($record){
                            return $record->wallet_histories->sum('price');
                        }),

                        Column::make("user.role")->heading("Request Date")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->request_date; }),
                        Column::make("user.status")->heading("Clearance Date")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->approved_date; }),
                        Column::make("user.last_login")->heading("Remarks")->formatStateUsing(function ($record){ return $record->wallet_histories[0]->remarks; }),
                    ])
                ])
            ]);
    }

}
