<?php

namespace App\Filament\Pages;

use App\Models\NewSite;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Infolists\Components\TextEntry;
use Filament\Pages\Page;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;
use Tapp\FilamentValueRangeFilter\Filters\ValueRangeFilter;

class DeletedSites extends Page implements HasForms,HasTable
{
    protected static ?string $navigationGroup = "Sites";
    use InteractsWithTable;
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-archive-box-x-mark';

    protected static string $view = 'filament.pages.deleted-sites';

    public function table(Table $table): Table
    {
        return $table
            ->query(
                NewSite::withCount("dupl")
                    ->where('delete_site','!=',null)
            )
            ->paginated(config('app.pagination'))
            ->filtersLayout(FiltersLayout::AboveContentCollapsible)
            ->columns([
                TextColumn::make('root_domain')->label("Root Domain")
                    ->formatStateUsing(function (NewSite $record){
                        if(Carbon::parse($record->created_at)->diffInDays(Carbon::now()) <= 30){
                            $text = "<span style='background: green;font-size: 11px;color: white;display: inline-block;margin-left: 3px;padding: 1px 7px;'>New Arrival</span>";
                        }else{
                            $text =  '';
                        }

                        return new HtmlString($record->root_domain. $text);
                    })->wrap()
                    ->description(function (NewSite $record) {
                        $text ='';
                        $text .= 'Category:- '.$record->category.'<br>';
                        $text .= 'Added on:- '.$record->created_at->toDayDateTimeString().'<br>';
                        if($record->new_order_process_detail){
                            $text .= 'Last Order Created :- '.Carbon::parse($record->new_order_process_detail->created_at)->toDayDateTimeString();
                        }

                        return new HtmlString('<span style="font-size: 12px;">'.$text.'</span>');
                    }),

                TextColumn::make("site_status")->label("Site Status")->formatStateUsing(function (NewSite $d) {
                    if($d->site_status == 0){
                        return new HtmlString('<span style="background: orange" class="bg-orange-500 text-white text-xs px-2 py-1">P</span>');
                    }elseif($d->site_status == 2) {
                        return new HtmlString('<span style="background: red" class="bg-red-500 text-white text-xs px-2 py-1" > R</span >');
                    }
                    elseif( $d->site_status == 1 ) {
                        return new HtmlString('<span style="background: green" class="bg-green-700 text-white text-xs px-2 py-1">A</span>');
                    }
                    return '';
                }),

                TextColumn::make('dr')->label("DR")->toggleable()->sortable(),
                TextColumn::make('da')->label("DA")->toggleable()->sortable(),
                TextColumn::make('rd')->label("RD")->toggleable()->sortable(),
                TextColumn::make('spam_score')->toggleable()->sortable(),

                TextColumn::make('traffic_source')->label("Traffic")->toggleable()->sortable(),
                TextColumn::make('gp_price')->label("GP Price")->toggleable()->sortable(),
                TextColumn::make('niche_edit_price')->label("Niche Price")->toggleable()->sortable(),
                TextColumn::make('fc_gp')->label("FC GP")->toggleable()->sortable(),
                TextColumn::make('fc_ne')->label("FC NE")->toggleable()->sortable(),
                TextColumn::make('website_status')->label("Website Status")->toggleable()->badge()->color(function (NewSite $d){
                    if($d->website_status == 'Acceptable'){
                        return 'success';
                    }
                    if($d->website_status == 'Non Acceptable'){
                        return 'danger';
                    }
                })->sortable(),
                TextColumn::make('email')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(function (NewSite $d) {
                        if(!empty($d->email)) {
                            return $d->email;
                        }else{
                            return optional($d->uploaded_user)->email;
                        }
                    }),
            ])
            ->filters([
                Filter::make("Filters")->form([
                    TextInput::make("root_domain"),
                    TextInput::make("category"),
                    TextInput::make("website_niche"),
                    Select::make("site_status")->live()->label("Status")
                        ->options([
                            "0"    =>  "Pending",
                            "1"    =>  "Approved",
                            "2"    =>  "Rejected",
                        ]),
                ])->columns(6)->columnSpanFull()->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when($data['root_domain'], function (Builder $query) use ($data) {
                            return $query->where('root_domain', 'LIKE','%'.$data['root_domain'].'%');
                        })
                        ->when($data['category'], function (Builder $query) use ($data) {
                            return $query->where('category', 'LIKE','%'.$data['category'].'%');
                        })
                        ->when($data['site_status'] != "", function (Builder $query) use ($data) {
                            return $query->where('site_status', $data['site_status']);
                        })
                        ->when($data['website_niche'], function (Builder $query) use ($data) {
                            return $query->where('website_niche', 'LIKE','%'.$data['website_niche'].'%');
                        });
                }),
                TernaryFilter::make('new_arrival')
                    ->label('New Arrival')
                    ->queries(
                        true: fn (Builder $query) => $query->where('created_at','>=',Carbon::now()->subDays(30)),
                        false: fn (Builder $query) => $query->where('created_at','<',Carbon::now()->subDays(30)),
                        blank: fn (Builder $query) => $query, // In this example, we do not want to filter the query when it is blank.
                    ),
                DateRangeFilter::make("created_at")->label("Added on"),
                ValueRangeFilter::make("dr")->label("DR"),
                ValueRangeFilter::make("da")->label("DA"),
                ValueRangeFilter::make("rd")->label("RD"),
                ValueRangeFilter::make("traffic_source")->label("Traffic"),
                ValueRangeFilter::make("gp_price"),
                ValueRangeFilter::make("niche_edit_price"),

            ])
            ->paginated(config('app.pagination'))
            ->actions([
                Action::make("view")->icon("heroicon-o-eye")->label(" ")->tooltip("View")->infolist([
                    TextEntry::make("website_niche")->inlineLabel(),
                    TextEntry::make("sample_url")->inlineLabel(),
                    TextEntry::make("href_url")->formatStateUsing(function (NewSite $d) {
                        return new HtmlString('<span style="word-wrap: anywhere; ">'.$d->href_url.'</span>');
                    }),
                    TextEntry::make("paypal_id")->inlineLabel(),
                    TextEntry::make("skype")->inlineLabel(),
                    TextEntry::make("whatsapp")->inlineLabel(),
                    TextEntry::make("country_source")->inlineLabel(),
                ]),
                Action::make("restore")->icon("heroicon-o-arrow-path")->label(" ")->tooltip("Restore")->color('success')->requiresConfirmation()->action(function (NewSite $record) {
                    try {
                        $record->delete_site = null;
                        $record->save();
                        DB::commit();
                    }catch (\Exception $exception){
                        DB::rollBack();
                       dd($exception->getMessage());
                    }
                }),
                Action::make("delete")->icon("heroicon-o-trash")->label(" ")->tooltip("Delete")->color('danger')->requiresConfirmation()->action(function (NewSite $record) {
                    $record->delete();
                }),
            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForWriterType()->askForFilename()->withColumns([
                        Column::make("root_domain"),
                        Column::make("category"),
                        Column::make("created_at")->heading('Added on'),
                        Column::make("site_status")->formatStateUsing(function ($state){
                            if($state == 0){
                                return 'Pending';
                            }elseif($state == 2) {
                                return 'Rejected';
                            }
                            elseif( $state == 1 ) {
                                return 'Approved';
                            }
                        }),
                        Column::make("dr"),
                        Column::make("da"),
                        Column::make("spam_score"),
                        Column::make("rd"),
                        Column::make("traffic_source"),
                        Column::make("gp_price"),
                        Column::make("niche_edit_price"),
                        Column::make("fc_gp"),
                        Column::make("fc_ne"),
                        Column::make("website_status"),
                        Column::make("email"),
                        Column::make("website_niche"),
                        Column::make("sample_url"),
                        Column::make("href_url"),
                        Column::make("paypal_id"),
                        Column::make("skype"),
                        Column::make("whatsapp"),
                        Column::make("country_source"),

                    ])
                ])
            ])
            ;
    }

}
