<?php

namespace App\Filament\Manager\Resources;

use App\Filament\Manager\Resources\RejectedOrdersFromBloggersResource\Pages;
use App\Filament\Manager\Resources\RejectedOrdersFromBloggersResource\RelationManagers;
use App\Models\NewOrder;
use App\Models\NewOrderProcessDetail;
use App\Models\RejectedOrdersFromBloggers;
use App\Project\NewOrderStatus;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;

class RejectedOrdersFromBloggersResource extends Resource
{
    protected static ?string $model = NewOrderProcessDetail::class;

    protected static ?string $label = "Bloggers";

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $navigationGroup = "Rejected Orders";

    public static function getNavigationBadge(): ?string
    {
        return bloggers_rejected_orders_for_manager();
    }


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(

                NewOrderProcessDetail::whereHas('new_order_process',function ($q){
                    $q->where('manager_id',Auth::id());
                })
                    ->where('status',NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION)
            )
            ->defaultSort('created_at', 'desc')
            ->recordUrl(null)
            ->paginated(config("app.pagination"))
            ->columns([
                Tables\Columns\TextColumn::make("new_order_process.new_order.order_id")->label("Order Id")->sortable(),
                Tables\Columns\TextColumn::make("new_order_process.new_order.order_type")->label("Order Type")->badge()->sortable()->color(function (NewOrderProcessDetail $record) {
                    if($record->new_order_process->new_order->order_type == "gp"){
                        return 'info';
                    }else{
                        return 'success';
                    }
                }),
                Tables\Columns\TextColumn::make("vendor.name")->label("Blogger")->toggleable()->sortable()->description(function (NewOrderProcessDetail $record) {
                    return $record->vendor->email;
                }),
                Tables\Columns\TextColumn::make("new_site.root_domain")->toggleable()->sortable()->label("Root Domain"),
                Tables\Columns\TextColumn::make("status")->toggleable()->sortable()
                    ->badge()
                    ->color(function (NewOrderProcessDetail $record) {
                        switch ($record->status) {
                            case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS:
                                return "success";
                            case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION:
                                return "danger";
                        }
                    })
                    ->formatStateUsing(function (NewOrderProcessDetail $record) {
                        switch ($record->status) {
                            case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS:
                                return "Accepted";
                            case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION:
                                return "Rejected";


                        }
                    }),

                Tables\Columns\TextColumn::make("reject_reason")->toggleable()->label("Reject Reason")->wrap(),
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                Tables\Filters\Filter::make("Search")->form([
                    Forms\Components\TextInput::make("order_id")->label("Order Id"),
                    Forms\Components\TextInput::make("user")->label("User name/email"),
                ]) ->query(function (Builder $query, array $data): Builder {
                    return $query

                        ->when($data['order_id'],function (Builder $query) use ($data){
                            return $query->whereHas("new_order_process.new_order",function ($query) use ($data){
                                return $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            });
                        })
                        ->when($data['user'],function (Builder $query) use ($data){
                            return $query->whereHas("vendor",function ($query) use ($data){
                                return $query->where('name','LIKE','%'.$data['user'].'%')->orWhere('email','LIKE','%'.$data['user'].'%');
                            });
                        })
                        ;
                })
                    ->columns(6)->columnSpanFull()
            ])
            ->actions([

            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->withColumns([
                        Column::make("new_order_process.new_order.order_id")->heading("Order Id"),
                        Column::make("vendor.name")->heading("Blogger Name"),
                        Column::make("vendor.email")->heading("Blogger Email"),
                        Column::make("new_site.root_domain")->heading("Root Domain"),
                        Column::make("status")->heading("Status")->formatStateUsing(function ($record){
                            switch ($record->status) {
                                case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_SUCCESS:
                                    return "Accepted";
                                case NewOrderStatus::$BLOGGER_PUSHED_BACK_TO_MANAGER_WITH_REJECTION:
                                    return "Rejected";


                            }
                        }),
                        Column::make("reject_reason")->heading("Rejection Reason"),

                    ])
                ])
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListRejectedOrdersFromBloggers::route('/'),
            'create' => Pages\CreateRejectedOrdersFromBloggers::route('/create'),
            'edit' => Pages\EditRejectedOrdersFromBloggers::route('/{record}/edit'),
        ];
    }
}
