<?php

namespace App\Filament\Clusters\Reporting\Pages;

use App\Filament\Clusters\Reporting;
use App\Models\OrdersView;
use App\Models\WalletHistory;
use Carbon\Carbon;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Pages\Page;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;

class BloggerReportPage extends Page implements HasForms,HasTable
{
    use InteractsWithTable;
    use InteractsWithForms;

    protected static ?string $navigationLabel = "Blogger Report";

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.clusters.reporting.pages.blogger-report-page';

    protected static ?string $cluster = Reporting::class;


    public function table(Table $table): Table
    {
        return $table
            ->query(function (){

                return WalletHistory::selectRaw("wallet_histories.id,sum(wallet_histories.price) as amount, new_sites.root_domain,wallet_histories.price,count(wallet_histories.price) as count,users.email,users.name")
                    ->join("withdraw_requests", "withdraw_requests.id", "wallet_histories.withdraw_request_id")
                    ->join("new_order_process_details", "new_order_process_details.id", "wallet_histories.order_detail_id")
                    ->join("new_sites", "new_sites.id", "new_order_process_details.new_site_id")
                    ->join("users", "users.id", "withdraw_requests.user_id")
                    ->where("withdraw_requests.status", 1)
                    ->groupBy(["new_sites.root_domain", "wallet_histories.price"]);

            })
            ->defaultSort("amount","desc")
            ->paginated(config("app.pagination"))
            ->columns([
                TextColumn::make("name")
                    ->copyable()
                    ->label("Blogger")->description(function ($record){
                    return $record->email;
                }),
                TextColumn::make("root_domain"),
                TextColumn::make("price"),
                TextColumn::make("count"),
                TextColumn::make("amount")->summarize(Sum::make()),
            ])

            ->actions([
                // ...
            ])
            ->filtersLayout(FiltersLayout::AboveContent)
            ->filters([
                Filter::make("search")->form([
                    TextInput::make("blogger"),
                    TextInput::make("root_domain"),

                    DatePicker::make("start_date")->default(function (){
                        return Carbon::today()->subMonth()->format("Y-m-d");
                    })->label("Start Date"),

                    DatePicker::make("end_date")
                        ->default(function (){
                            return Carbon::today()->format("Y-m-d");
                        })
                        ->label("End Date"),

                ])
                    ->query(function (\Illuminate\Database\Eloquent\Builder $query, array $data){
                        return $query
                            ->when($data['start_date'] && $data['end_date'], function (Builder $query) use ($data){
                                $query->whereBetween('wallet_histories.approved_date', [$data["start_date"].' 00:00:00', $data["end_date"].' 23:59:59' ]);
                            })
                            ->when($data['blogger'], function (Builder $query) use ($data){
                                $query->where("users.name", "like", "%".$data["blogger"]."%")
                                    ->orWhere("users.email", "like", "%".$data["blogger"]."%");
                            })
                            ->when($data['root_domain'], function (Builder $query) use ($data){
                                $query->where("new_sites.root_domain", "like", "%".$data["root_domain"]."%");
                            })
                            ;

                    })
                    ->columns(4)->columnSpanFull()
            ])
            ->bulkActions([

            ]);
    }

}
