<?php

namespace App\Filament\Blogger\Resources;

use App\Filament\Blogger\Resources\WalletResource\Pages;
use App\Filament\Blogger\Resources\WalletResource\RelationManagers;
use App\Models\Wallet;
use App\Models\WalletHistory;
use Carbon\Carbon;
use Filament\Forms;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;

class WalletResource extends Resource
{
    protected static ?string $navigationGroup = "Payments";
    protected static ?string $model = WalletHistory::class;

    protected static ?string $label  = "Wallet";

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function getLabel(): ?string
    {
        return "Wallet (Balance:- ".vendor_balance().")";
    }


    public static function table(Table $table): Table
    {
        return $table
            ->query(
                WalletHistory::has("new_order_process_detail")->with("new_order_process_detail.new_order_process.new_order")->whereHas('wallet',function ($q){
                    $q->where('user_id',Auth::id());
                })
            )
            ->defaultSort('id','desc')
            ->recordUrl(null)
            ->columns([
                Tables\Columns\TextColumn::make("created_at")->label("Date"),
                Tables\Columns\TextColumn::make("new_order_process_detail.new_order_process.new_order.order_id")
                    ->toggleable()
                    ->label("Order Id")
                    ->sortable()
                    ->badge()
                    ->wrap()
                    ->description(function ($record) {
                        return optional($record->new_order_process_detail)->submit_url;
                    }),
                Tables\Columns\TextColumn::make("remarks")
                    ->toggleable()
                    ->label("Remarks")
                    ->wrap()
                    ->toggleable(isToggledHiddenByDefault: true)
                ,
                Tables\Columns\TextColumn::make("price")->toggleable()->label("Credit")->sortable(),
                Tables\Columns\TextColumn::make("status")->toggleable()->label("Withdrawal Status")
                    ->badge()->color("warning")
                    ->formatStateUsing(function ($record){
                        if($record->status == 0 && $record->request_date ){
                            return 'Pending';
                        }
                    })->description(function ($record){
                        if($record->status == 0 && $record->request_date ){
                            return Carbon::parse($record->request_date)->toDayDateTimeString();
                        }
                    }),
                Tables\Columns\TextColumn::make("approved_date")
                    ->toggleable()
                    ->badge()
                    ->label("Approved Status")
                    ->formatStateUsing(function ($record){
                        if($record->status == 1 && $record->approved_date ){
                            return 'Approved';
                        }
                    })->description(function ($record){
                        if($record->status == 1 && $record->approved_date ){
                            return Carbon::parse($record->approved_date)->toDayDateTimeString();
                        }
                    }),
            ])
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->filters([
                DateRangeFilter::make("created_at")->label("Search By Date"),
                Tables\Filters\Filter::make("Search")->form([
                    TextInput::make("order_id")->label("Order Id"),
                    TextInput::make("submit_url")->label("Search by Submit URL"),
                ])->query(function (Builder $query,$data){
                    return $query
                        ->when($data['order_id'],function (Builder $query) use ($data) {
                            $query->whereHas("new_order_process_detail.new_order_process.new_order",function (Builder $query) use ($data){
                               $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            });
                        })
                        ->when($data['submit_url'],function (Builder $query) use ($data) {
                            $query->whereHas("new_order_process_detail",function (Builder $query) use ($data){
                                $query->where('submit_url','LIKE','%'.$data['submit_url'].'%');
                            });
                        })
                        ;
                })->columnSpan(4)->columns(4)
            ])
            ->actions([
            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListWallets::route('/'),
            'create' => Pages\CreateWallet::route('/create'),
            'edit' => Pages\EditWallet::route('/{record}/edit'),
        ];
    }
}
