<?php

namespace App\Filament\Blogger\Pages;

use App\Models\BulkUpload;
use App\Models\NewSite;
use App\Project\Site\NewSiteStatus;
use Carbon\Carbon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Concerns\InteractsWithInfolists;
use Filament\Infolists\Contracts\HasInfolists;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Support\Enums\FontWeight;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;
use PHPUnit\Exception;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;
use pxlrbt\FilamentExcel\Columns\Column;
use pxlrbt\FilamentExcel\Exports\ExcelExport;
use Tapp\FilamentValueRangeFilter\Filters\ValueRangeFilter;

class ViewUploadedBulkSites extends Page implements HasForms, HasTable, HasInfolists
{
    use InteractsWithForms;
    use InteractsWithTable;
    use InteractsWithInfolists;

    protected static ?string $navigationGroup = "Bulk Sites";
    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    public $bulk_upload_status;
    protected static string $view = 'filament.blogger.pages.view-uploaded-bulk-sites';

    public function mount(): void
    {
        $this->bulk_upload_status =
            BulkUpload::where('vendor_id',Auth::id())
                ->where(function ($q){
                    $q->where('status',0)->orWhere('status',1);
                })
                ->first();
    }


    public function productInfolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->record($this->bulk_upload_status)
            ->schema([
                Section::make("Details")
                    ->schema([
                        TextEntry::make("created_at")->label("Uploaded Date Time")->inlineLabel()->weight(FontWeight::Bold),
                        TextEntry::make("status")->label("Status")->inlineLabel()->weight(FontWeight::Bold)
                            ->formatStateUsing(function ($record){
                                if($record->status == 0 ){
                                    return 'Waiting for Admin Approval';
                                }
                                if( $record->status == 1 ){
                                    return 'Admin Approved and Uploaded your sites. Now check uploaded sites to accept or decline';
                                }
                            })

                    ])
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(
                NewSite::where('bulk_upload_id',$this->bulk_upload_status ? $this->bulk_upload_status->id : null)
            )
            ->defaultSort('id','desc')
            ->striped()
            ->recordUrl(null)
            ->paginated(config('app.pagination'))
            ->columns([
                TextColumn::make('root_domain')->label("Root Domain")
                    ->formatStateUsing(function (NewSite $record){
                        if(Carbon::parse($record->created_at)->diffInDays(Carbon::now()) <= 30){
                            $text = "<span style='background: green;font-size: 11px;color: white;display: inline-block;margin-left: 3px;padding: 1px 7px;'>New Arrival</span>";
                        }else{
                            $text =  '';
                        }

                        return new HtmlString($record->root_domain. $text);
                    })->wrap()
                    ->description(function (NewSite $record) {
                        $text ='';
                        $text .= 'Category:- '.$record->category.'<br>';
                        $text .= 'Added on:- '.$record->created_at->toDayDateTimeString().'<br>';


                        return new HtmlString('<span style="font-size: 12px;">'.$text.'</span>');
                    }),

                TextColumn::make("site_status")->label("Site Status")->formatStateUsing(function (NewSite $d) {
                    if($d->site_status == 0){
                        return new HtmlString('<span style="background: orange" class="bg-orange-500 text-white text-xs px-2 py-1">P</span>');
                    }elseif($d->site_status == 2) {
                        return new HtmlString('<span style="background: red" class="bg-red-500 text-white text-xs px-2 py-1" > R</span >');
                    }
                    elseif( $d->site_status == 1 ) {
                        return new HtmlString('<span style="background: green" class="bg-green-700 text-white text-xs px-2 py-1">A</span>');
                    }
                    return '';
                }),


                TextColumn::make('dr')->label("DR")->sortable(),
                TextColumn::make('da')->label("DA")->sortable(),
                TextColumn::make('rd')->label("RD")->sortable(),
                TextColumn::make('spam_score')->sortable(),

                TextColumn::make('traffic_source')->label("Traffic")->sortable(),
                TextColumn::make('gp_price')->label("GP Price")->sortable(),
                TextColumn::make('niche_edit_price')->label("Niche Price")->sortable(),
                TextColumn::make('fc_gp')->label("FC GP")->sortable(),
                TextColumn::make('fc_ne')->label("FC NE")->sortable(),
                TextColumn::make('website_status')->label("Website Status")->badge()->color(function (NewSite $d){
                    if($d->website_status == 'Acceptable'){
                        return 'success';
                    }
                    if($d->website_status == 'Non Acceptable'){
                        return 'danger';
                    }
                })->sortable(),
                TextColumn::make('email')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(function (NewSite $d) {
                        if(!empty($d->email)) {
                            return $d->email;
                        }else{
                            return optional($d->uploaded_user)->email;
                        }
                    }),

            ])
            ->filtersLayout(FiltersLayout::AboveContentCollapsible)
            ->filters([
                Filter::make("Filters")->form([
                    TextInput::make("category"),
                    TextInput::make("root_domain"),
                    TextInput::make("website_niche"),
                    Select::make("site_status")->live()->label("Status")
                        ->options([
                            "0"    =>  "Pending",
                            "1"    =>  "Approved",
                            "2"    =>  "Rejected",
                        ]),
                ])->columns(6)->columnSpanFull()->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when($data['category'], function (Builder $query) use ($data) {
                            return $query->where('category', 'LIKE','%'.$data['category'].'%');
                        })
                        ->when($data['root_domain'], function (Builder $query) use ($data) {
                            return $query->where('root_domain', 'LIKE','%'.$data['root_domain'].'%');
                        })
                        ->when($data['site_status'] != "", function (Builder $query) use ($data) {
                            return $query->where('site_status', $data['site_status']);
                        })
                        ->when($data['website_niche'], function (Builder $query) use ($data) {
                            return $query->where('website_niche', 'LIKE','%'.$data['website_niche'].'%');
                        });
                }),
                TernaryFilter::make('new_arrival')
                    ->label('New Arrival')
                    ->queries(
                        true: fn (Builder $query) => $query->where('created_at','>=',Carbon::now()->subDays(30)),
                        false: fn (Builder $query) => $query->where('created_at','<',Carbon::now()->subDays(30)),
                        blank: fn (Builder $query) => $query, // In this example, we do not want to filter the query when it is blank.
                    ),
                DateRangeFilter::make("created_at")->label("Added on"),
                ValueRangeFilter::make("dr")->label("DR"),
                ValueRangeFilter::make("da")->label("DA"),
                ValueRangeFilter::make("rd")->label("RD"),
                ValueRangeFilter::make("traffic_source")->label("Traffic"),
                ValueRangeFilter::make("gp_price"),
                ValueRangeFilter::make("niche_edit_price"),

            ])
            ->actions([
                Action::make("view")->icon("heroicon-m-eye")->label(" ")->tooltip("view")->infolist([
                    TextEntry::make("website_niche")->inlineLabel(),
                    TextEntry::make("sample_url")->inlineLabel(),
                    TextEntry::make("href_url")->formatStateUsing(function (NewSite $d) {
                        return new HtmlString('<span style="word-wrap: anywhere; ">'.$d->href_url.'</span>');
                    }),
                    TextEntry::make("paypal_id")->inlineLabel(),
                    TextEntry::make("skype")->inlineLabel(),
                    TextEntry::make("whatsapp")->inlineLabel(),
                    TextEntry::make("country_source")->inlineLabel(),


                ]),

                ActionGroup::make([
                    Action::make("Approve")
                        ->icon('heroicon-m-star')
                        ->color('success')
                        ->action(function (NewSite $record) {
                            try{
                                DB::beginTransaction();
                                $bulk_upload_site = $record;




                                $bulk_upload_site->site_status = NewSiteStatus::APPROVED;
                                $bulk_upload_site->bulk_upload_id = null;
                                $bulk_upload_site->bulk_upload_status = null;
                                $bulk_upload_site->save();

                                $bulk_upload= BulkUpload::withCount("bulk_upload_sites")->where('vendor_id',Auth::id())
                                    ->where('status',1)
                                    ->firstOrFail();



                                if( $bulk_upload->bulk_upload_sites_count == 0 ){
                                    $bulk_upload->delete();
                                }

                                DB::commit();
                                Notification::make("success")->title("Site Approved")->success()->send();
                            }catch (Exception $exception){
                                Notification::make("error")->title($exception->getMessage())->danger()->send();
                                    dd($exception->getMessage());
                            }
                        }),
                    Action::make("Delete")
                        ->icon('heroicon-m-x-mark')
                        ->color('danger')
                        ->action(function (NewSite $record) {
                            try{

                                DB::beginTransaction();
                                $bulk_upload_site = $record;




                                $bulk_upload_site->site_status = NewSiteStatus::APPROVED;
                                $bulk_upload_site->bulk_upload_id = null;
                                $bulk_upload_site->bulk_upload_status = null;
                                $bulk_upload_site->delete_site = 1;
                                $bulk_upload_site->save();

                                $bulk_upload= BulkUpload::withCount("bulk_upload_sites")->where('vendor_id',Auth::id())
                                    ->where('status',1)
                                    ->firstOrFail();



                                if( $bulk_upload->bulk_upload_sites_count == 0 ){
                                    $bulk_upload->delete();
                                }

                                DB::commit();
                                Notification::make("success")->title("Site Deleted")->success()->send();
                            }catch (Exception $exception){
                                Notification::make("error")->title($exception->getMessage())->danger()->send();
                            }
                        }),

                ])

            ])
            ->bulkActions([
                ExportBulkAction::make()->exports([
                    ExcelExport::make()->askForWriterType()->askForFilename()->withColumns([
                        Column::make("root_domain"),
                        Column::make("category"),
                        Column::make("created_at")->heading('Added on'),
                        Column::make("site_status")->formatStateUsing(function ($state){
                            if($state == 0){
                                return 'Pending';
                            }elseif($state == 2) {
                                return 'Rejected';
                            }
                            elseif( $state == 1 ) {
                                return 'Approved';
                            }
                        }),
                        Column::make("dr"),
                        Column::make("da"),
                        Column::make("spam_score"),
                        Column::make("rd"),
                        Column::make("traffic_source"),
                        Column::make("gp_price"),
                        Column::make("niche_edit_price"),
                        Column::make("fc_gp"),
                        Column::make("fc_ne"),
                        Column::make("website_status"),
                        Column::make("email"),
                        Column::make("website_niche"),
                        Column::make("sample_url"),
                        Column::make("href_url"),
                        Column::make("paypal_id"),
                        Column::make("skype"),
                        Column::make("whatsapp"),
                        Column::make("country_source"),

                    ])
                ])
            ])
            ;
    }

}
