@extends('layouts.shop')

@section('title', 'Order #' . $order->order_number)

@push('head_scripts')
    @if($order->status === 'shipped' && $order->deliveryAgent)
        <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css"
            integrity="sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY=" crossorigin="" />
        <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"
            integrity="sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo=" crossorigin=""></script>
    @endif
@endpush

@section('content')
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
        <!-- Breadcrumbs -->
        <nav class="flex mb-6" aria-label="Breadcrumb">
            <ol class="flex items-center space-x-2">
                <li>
                    <a href="{{ route('user.orders') }}"
                        class="text-sm font-medium text-gray-500 hover:text-indigo-600 transition-colors">Orders</a>
                </li>
                <li class="flex items-center">
                    <svg class="h-4 w-4 text-gray-400" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd"
                            d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z"
                            clip-rule="evenodd" />
                    </svg>
                    <span class="ml-2 text-sm font-bold text-indigo-600">Order #{{ $order->order_number }}</span>
                </li>
            </ol>
        </nav>

        <!-- Header Section -->
        <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden mb-8">
            <div
                class="px-6 py-8 sm:px-10 bg-gradient-to-r from-indigo-50 via-white to-indigo-50 border-b border-gray-100 flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                <div>
                    <h1 class="text-3xl font-extrabold text-gray-900 tracking-tight">Order Details</h1>
                    <div class="mt-2 flex items-center gap-3">
                        <span
                            class="inline-flex items-center px-3 py-1 rounded-full text-sm font-bold bg-indigo-100 text-indigo-700">
                            #{{ $order->order_number }}
                        </span>
                        <span class="text-sm text-gray-500 font-medium">Placed on {{ $order->created_at->format('M d, Y') }}
                            at {{ $order->created_at->format('h:i A') }}</span>
                    </div>
                </div>

                <div class="flex flex-wrap items-center gap-3">
                    <form action="{{ route('user.orders.reorder', $order->id) }}" method="POST">
                        @csrf
                        <button type="submit"
                            class="inline-flex items-center px-4 py-2 border border-indigo-200 text-sm font-bold rounded-xl text-indigo-600 bg-white hover:bg-indigo-50 transition-all duration-200 shadow-sm">
                            <svg class="h-4 w-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                            </svg>
                            Buy Again
                        </button>
                    </form>

                    <a href="{{ route('user.orders.invoice', $order->id) }}"
                        class="inline-flex items-center px-4 py-2 border border-gray-200 text-sm font-bold rounded-xl text-gray-700 bg-white hover:bg-gray-50 transition-all duration-200 shadow-sm">
                        <svg class="h-4 w-4 mr-2 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
                        </svg>
                        Invoice
                    </a>

                    @php
                        $statusLabel = ucfirst($order->status);
                        $statusColor = match ($order->status) {
                            'completed' => 'bg-green-500',
                            'processing' => 'bg-blue-500',
                            'cancelled' => 'bg-red-500',
                            'shipped' => 'bg-indigo-500',
                            default => 'bg-yellow-500'
                        };

                        if ($order->returnRequest) {
                            $rStatus = $order->returnRequest->status;
                            $statusLabel = 'Return ' . ucfirst(str_replace('_', ' ', $rStatus));
                            if ($rStatus == 'requested')
                                $statusColor = 'bg-yellow-500';
                            elseif ($rStatus == 'approved')
                                $statusColor = 'bg-blue-500';
                            elseif ($rStatus == 'rejected')
                                $statusColor = 'bg-red-500';
                            elseif ($rStatus == 'received')
                                $statusColor = 'bg-purple-500';
                            elseif ($rStatus == 'refund_initiated')
                                $statusColor = 'bg-indigo-500';
                            elseif ($rStatus == 'refunded')
                                $statusColor = 'bg-green-500';
                        }
                    @endphp
                    <span
                        class="inline-flex items-center px-4 py-2 rounded-xl text-sm font-bold text-white shadow-md {{ $statusColor }}">
                        {{ $statusLabel }}
                    </span>
                </div>
            </div>

            <!-- Inner Content (Timeline & Info) -->
            <div class="p-6 sm:p-10">
                <!-- Order Timeline -->
                @php
                    $steps = [
                        ['status' => 'pending', 'label' => 'Placed', 'icon' => 'M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z'],
                        ['status' => 'processing', 'label' => 'Processed', 'icon' => 'M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4'],
                        ['status' => 'shipped', 'label' => 'In Transit', 'icon' => 'M9 17a2 2 0 11-4 0 2 2 0 014 0zM19 17a2 2 0 11-4 0 2 2 0 014 0z M13 16V6a1 1 0 00-1-1H4a1 1 0 00-1 1v10a1 1 0 001 1h1m8-1a1 1 0 011 1v2a1 1 0 01-1 1m0-4h3m-11 0h1m11 0a1 1 0 001-1V7a1 1 0 00-1-1h-3.3a1 1 0 00-1 1v9h3.3z'],
                        ['status' => 'completed', 'label' => 'Delivered', 'icon' => 'M5 13l4 4L19 7'],
                    ];
                    $histories = $order->statusHistories->keyBy('status');
                    $isCancelled = $order->status === 'cancelled';

                    if ($order->returnRequest) {
                        $ret = $order->returnRequest;

                        // Add Return Requested Step
                        $steps[] = ['status' => 'return_requested', 'label' => 'Return Req.', 'icon' => 'M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6'];
                        // Use created_at for request time
                        $histories->put('return_requested', $ret);

                        if (in_array($ret->status, ['approved', 'received', 'refund_initiated', 'refunded'])) {
                            $steps[] = ['status' => 'return_approved', 'label' => 'Return Appr.', 'icon' => 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'];
                            // Use updated_at as proxy for approval time if active, else it's approximate
                            $histories->put('return_approved', (object) ['created_at' => $ret->updated_at]);
                        }

                        if ($ret->status == 'rejected') {
                            $steps[] = ['status' => 'return_rejected', 'label' => 'Return Rejected', 'icon' => 'M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z'];
                            $histories->put('return_rejected', (object) ['created_at' => $ret->updated_at]);
                        }

                        if ($ret->status == 'refunded') {
                            $steps[] = ['status' => 'refunded', 'label' => 'Refunded', 'icon' => 'M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z'];
                            $histories->put('refunded', (object) ['created_at' => $ret->refunded_at]);
                        }
                    }

                    $furthestStep = -1;
                    foreach ($steps as $index => $step) {
                        if ($histories->has($step['status']))
                            $furthestStep = $index;
                    }
                @endphp

                @php
                    $totalShownSteps = count($steps) + ($isCancelled ? 1 : 0);
                    $progressPercentage = ($totalShownSteps > 1) ? ($furthestStep / ($totalShownSteps - 1)) : 0;

                    $statusLabelColors = [
                        'pending' => 'text-indigo-600',
                        'processing' => 'text-blue-600',
                        'shipped' => 'text-amber-600',
                        'completed' => 'text-green-600',
                    ];
                @endphp

                <div class="mb-14">
                    <div class="relative">
                        <!-- Progress Bar Background (Center-to-Center) -->
                        <div class="absolute top-6 left-6 right-6 h-1 bg-gray-100 rounded-full"></div>

                        <!-- Progress Bar Filled -->
                        @if($furthestStep >= 0 && !$isCancelled)
                            <div class="absolute top-6 left-6 h-1 bg-gradient-to-r from-indigo-500 to-blue-500 transition-all duration-1000 rounded-full shadow-[0_0_10px_rgba(99,102,241,0.5)]"
                                style="width: calc({{ $progressPercentage * 100 }}% - {{ $progressPercentage * 48 }}px)"></div>
                        @endif

                        <div class="flex justify-between relative z-10 text-center">
                            @foreach ($steps as $index => $step)
                                                @php
                                                    $isActive = $histories->has($step['status']);
                                                    $isCurrent = $order->status === $step['status'];
                                                    $history = $histories->get($step['status']);
                                                @endphp
                                                <div class="flex flex-col items-center flex-1">
                                                    <div class="w-12 h-12 rounded-2xl flex flex-shrink-0 items-center justify-center border-2 transition-all duration-500
                                                                                                                                                                                                                                                                                                                    {{ $isActive
                                ? ($isCancelled && $isCurrent ? 'bg-red-500 border-red-500 text-white shadow-lg' : 'bg-indigo-600 border-indigo-600 text-white shadow-lg scale-110')
                                : 'bg-white border-gray-200 text-gray-400' }}">
                                                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                                d="{{ $step['icon'] }}"></path>
                                                        </svg>
                                                    </div>
                                                    <div class="mt-4 px-1">
                                                        <span
                                                            class="block text-[10px] sm:text-xs font-black uppercase tracking-widest {{ $isActive ? ($statusLabelColors[$step['status']] ?? 'text-gray-900') : 'text-gray-400' }}">
                                                            {{ $step['label'] }}
                                                        </span>
                                                        @if($isActive && $history)
                                                            <p class="hidden sm:block text-[10px] text-gray-500 mt-1 font-bold">
                                                                {{ $history->created_at->format('M d') }}<br>
                                                                {{ $history->created_at->format('h:i A') }}
                                                            </p>
                                                        @endif
                                                    </div>
                                                </div>
                            @endforeach

                            @if($isCancelled)
                                <div class="flex flex-col items-center flex-1">
                                    <div
                                        class="w-12 h-12 rounded-2xl flex flex-shrink-0 items-center justify-center border-2 bg-red-600 border-red-600 text-white shadow-lg scale-110 animate-pulse">
                                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </div>
                                    <div class="mt-4 px-1">
                                        <span
                                            class="block text-[10px] sm:text-xs font-black uppercase tracking-widest text-red-600">
                                            Cancelled
                                        </span>
                                        @php $cancelHistory = $histories->get('cancelled'); @endphp
                                        @if($cancelHistory)
                                            <p class="hidden sm:block text-[10px] text-red-500 mt-1 font-bold">
                                                {{ $cancelHistory->created_at->format('M d') }}<br>
                                                {{ $cancelHistory->created_at->format('h:i A') }}
                                            </p>
                                        @endif
                                    </div>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-8 mt-10">
                    <!-- Shipping -->
                    <div
                        class="bg-slate-50 rounded-2xl p-6 border border-slate-100 relative group transition-colors hover:bg-slate-100/50">
                        <div
                            class="absolute -top-3 left-6 px-3 py-1 bg-white border border-slate-200 rounded-full text-[10px] font-black uppercase tracking-widest text-slate-500">
                            Shipping To</div>
                        <div class="flex items-start gap-4">
                            <div class="p-3 bg-white rounded-xl shadow-sm text-indigo-600">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                                </svg>
                            </div>
                            <div>
                                @if($order->shippingAddress)
                                    <p class="text-gray-900 font-extrabold mb-1">{{ $order->shippingAddress->name }}</p>
                                    <p class="text-sm text-gray-600 leading-relaxed">
                                        {{ $order->shippingAddress->address_line1 }}<br>
                                        @if($order->shippingAddress->address_line2)
                                        {{ $order->shippingAddress->address_line2 }}<br> @endif
                                        {{ $order->shippingAddress->city }}, {{ $order->shippingAddress->state }} -
                                        {{ $order->shippingAddress->postal_code }}<br>
                                        {{ $order->shippingAddress->country }}<br>
                                        <span class="block mt-2 font-bold text-gray-700">Phone:
                                            {{ $order->shippingAddress->phone }}</span>
                                    </p>
                                @else
                                    <p class="text-sm text-gray-400 italic">No shipping details provided.</p>
                                @endif
                            </div>
                        </div>
                    </div>

                    <!-- Payment -->
                    <div
                        class="bg-slate-50 rounded-2xl p-6 border border-slate-100 relative group transition-colors hover:bg-slate-100/50">
                        <div
                            class="absolute -top-3 left-6 px-3 py-1 bg-white border border-slate-200 rounded-full text-[10px] font-black uppercase tracking-widest text-slate-500">
                            Payment Info</div>
                        <div class="flex items-start gap-4">
                            <div class="p-3 bg-white rounded-xl shadow-sm text-indigo-600">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                                </svg>
                            </div>
                            <div class="flex-1">
                                <p class="text-gray-900 font-extrabold mb-1 capitalize">
                                    {{ str_replace('_', ' ', $order->payment_method ?? 'Not selected') }}
                                </p>
                                <span
                                    class="px-2 py-0.5 bg-indigo-100 text-indigo-700 text-[10px] font-black rounded-md uppercase tracking-wide">
                                    {{ ucfirst($order->payment_status) }}
                                </span>

                                @if($order->payment_method == 'razorpay' || $order->payment_method == 'phonepe')
                                    <div class="mt-4 pt-4 border-t border-gray-200/60 divide-y divide-gray-100">
                                        @if($order->transaction_id)
                                            <div class="flex justify-between py-1.5">
                                                <span
                                                    class="text-[10px] font-bold text-gray-400 uppercase tracking-widest">Transaction
                                                    ID</span>
                                                <span
                                                    class="text-xs font-mono text-gray-600 break-all ml-4">{{ $order->transaction_id }}</span>
                                            </div>
                                        @endif
                                    </div>
                                @endif

                                @if($order->status == 'cancelled' && $order->cancellation_reason)
                                    <div class="mt-4 p-3 bg-red-50 rounded-xl border border-red-100">
                                        <p class="text-[10px] font-black text-red-600 uppercase tracking-widest mb-1">Reason for
                                            Cancellation</p>
                                        <p class="text-sm text-red-800 font-medium italic">"{{ $order->cancellation_reason }}"
                                        </p>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>

                @if($order->deliveryAgent)
                    <div class="grid grid-cols-1 mt-8">
                        <div
                            class="bg-indigo-50 rounded-2xl p-6 border border-indigo-100 relative group transition-colors hover:bg-indigo-100/50">
                            <div
                                class="absolute -top-3 left-6 px-3 py-1 bg-white border border-indigo-200 rounded-full text-[10px] font-black uppercase tracking-widest text-indigo-500">
                                Delivery Executive
                            </div>
                            <div class="flex items-center justify-between flex-wrap gap-4">
                                <div class="flex items-start gap-4">
                                    <div class="p-3 bg-white rounded-xl shadow-sm text-indigo-600">
                                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                        </svg>
                                    </div>
                                    <div>
                                        <p class="text-gray-900 font-extrabold mb-1">{{ $order->deliveryAgent->user->name }}</p>
                                        <p class="text-sm text-gray-600 font-medium">Lugani Verified Delivery Partner</p>
                                    </div>
                                </div>

                                @if($order->order_scope === 'Local' && $order->status === 'shipped')
                                    <a href="tel:{{ $order->deliveryAgent->phone }}"
                                        class="inline-flex items-center px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-bold rounded-xl shadow-lg shadow-green-100 transition-all active:scale-95 text-sm flex-shrink-0">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                                        </svg>
                                        Call Executive
                                    </a>
                                @endif
                            </div>
                        </div>
                    </div>
                @endif
            </div>
        </div>

        @if($order->status === 'shipped' && $order->deliveryAgent)
            <!-- Live Tracking Map -->
            <div class="bg-white rounded-3xl shadow-sm border border-gray-100 overflow-hidden mb-8">
                <div class="px-6 py-4 border-b border-gray-100 bg-gray-50 flex justify-between items-center">
                    <div class="flex items-center gap-3">
                        <div class="w-2 h-2 rounded-full bg-green-500 animate-pulse"></div>
                        <h3 class="text-lg font-black text-gray-900 tracking-tight">Live Delivery Tracking</h3>
                    </div>
                    <span class="text-[10px] font-black text-gray-400 uppercase tracking-widest">Updating every 10s</span>
                </div>
                <div id="map" class="w-full relative"
                    style="height: 500px; min-height: 500px; background: #f8fafc; border: 1px solid #e2e8f0; z-index: 1;"></div>
                <div class="p-4 bg-indigo-50 border-t border-indigo-100 flex items-center justify-between">
                    <div class="flex items-center gap-3">
                        <div class="p-2 bg-white rounded-lg shadow-sm">
                            <svg class="w-5 h-5 text-indigo-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                            </svg>
                        </div>
                        <div>
                            <p class="text-xs font-black text-indigo-900 uppercase tracking-wide">Current Location</p>
                            <p class="text-sm text-indigo-700 font-bold" id="last-update">Waiting for signal...</p>
                        </div>
                    </div>
                </div>
            </div>
        @endif

        <!-- Rating Section -->
        @if($order->status === 'completed')
            <div class="mt-12 bg-white rounded-3xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="px-6 py-8 sm:px-10 border-b border-gray-100 bg-slate-50/50">
                    <h2 class="text-2xl font-black text-gray-900 tracking-tight">
                        {{ $order->productReviews->isNotEmpty() ? 'Your Feedback' : 'Share Your Experience' }}
                    </h2>
                    <p class="text-sm text-gray-500 font-medium mt-1">
                        {{ $order->productReviews->isNotEmpty() ? 'Thank you for rating your experience!' : 'Rate your products and the delivery executive' }}
                    </p>
                </div>

                @if($order->productReviews->isEmpty())
                    <form action="{{ route('user.orders.rate', $order->id) }}" method="POST" class="p-6 sm:p-10 space-y-10">
                        @csrf

                        <!-- Product Ratings -->
                        <div class="space-y-8">
                            @foreach($order->items as $index => $item)
                                <div class="flex flex-col md:flex-row gap-6 pb-8 {{ !$loop->last ? 'border-b border-gray-100' : '' }}">
                                    <div class="flex-shrink-0 h-16 w-16 bg-gray-50 rounded-xl overflow-hidden border border-gray-100">
                                        @if($item->variation && $item->variation->image_path)
                                            <img class="h-full w-full object-cover" src="{{ $item->variation->image_path }}" alt="">
                                        @elseif($item->product->images->isNotEmpty())
                                            <img class="h-full w-full object-cover" src="{{ $item->product->images->first()->image_path }}"
                                                alt="">
                                        @endif
                                    </div>
                                    <div class="flex-1">
                                        <h4 class="text-base font-black text-gray-900 mb-4">{{ $item->product->name }}</h4>
                                        <input type="hidden" name="product_ratings[{{ $index }}][id]" value="{{ $item->product_id }}">

                                        <div class="flex flex-col gap-4">
                                            <div class="flex items-center gap-1" x-data="{ rating: 0, hover: 0 }">
                                                <template x-for="i in [1, 2, 3, 4, 5]" :key="i">
                                                    <button type="button" @click="rating = i" @mouseenter="hover = i"
                                                        @mouseleave="hover = 0"
                                                        class="group focus:outline-none transition-all duration-200 transform hover:scale-110">
                                                        <svg class="w-10 h-10 transition-all duration-200"
                                                            :class="(hover >= i || (hover === 0 && rating >= i)) ? 'fill-amber-500 text-amber-500 scale-110 drop-shadow-sm' : 'fill-gray-200 text-gray-300'"
                                                            viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                                                            <path
                                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                        </svg>
                                                    </button>
                                                </template>
                                                <input type="hidden" name="product_ratings[{{ $index }}][rating]" :value="rating">
                                            </div>
                                            <textarea name="product_ratings[{{ $index }}][comment]" rows="2"
                                                class="block w-full border-2 border-gray-100 focus:border-indigo-500 focus:ring-indigo-500 rounded-2xl p-4 text-sm font-medium bg-slate-50/50"
                                                placeholder="Tell us more about the product..."></textarea>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </div>

                        @if($order->deliveryAgent && !$order->agentRating)
                            <!-- Agent Rating -->
                            <div class="pt-8 border-t-2 border-dashed border-gray-100">
                                <div class="bg-indigo-50/50 rounded-3xl p-6 sm:p-10 border border-indigo-100">
                                    <div class="flex items-center gap-4 mb-8">
                                        <div
                                            class="h-14 w-14 bg-white rounded-full flex items-center justify-center text-indigo-600 shadow-sm">
                                            <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-xl font-black text-gray-900 leading-tight">Rate
                                                {{ $order->deliveryAgent->user->name }}
                                            </h4>
                                            <p class="text-sm text-gray-500 font-medium">How was your delivery experience?</p>
                                        </div>
                                    </div>

                                    <div class="flex flex-col gap-6" x-data="{ rating: 0, hover: 0 }">
                                        <div class="flex items-center gap-2">
                                            <template x-for="i in [1, 2, 3, 4, 5]" :key="i">
                                                <button type="button" @click="rating = i" @mouseenter="hover = i"
                                                    @mouseleave="hover = 0"
                                                    class="group focus:outline-none transition-all duration-200 transform hover:scale-110">
                                                    <svg class="w-12 h-12 transition-all duration-200"
                                                        :class="(hover >= i || (hover === 0 && rating >= i)) ? 'fill-amber-500 text-amber-500 scale-110 drop-shadow-md' : 'fill-gray-200 text-gray-300'"
                                                        viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                                                        <path
                                                            d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                    </svg>
                                                </button>
                                            </template>
                                            <input type="hidden" name="agent_rating" :value="rating">
                                        </div>
                                        <textarea name="agent_comment" rows="3"
                                            class="block w-full border-2 border-gray-100 focus:border-indigo-500 focus:ring-indigo-500 rounded-2xl p-4 text-sm font-medium bg-white"
                                            placeholder="Any feedback for the delivery executive?"></textarea>
                                    </div>
                                </div>
                            </div>
                        @endif

                        <!-- Submit Review Button -->
                        <div class="flex justify-end pt-6">
                            <button type="submit"
                                class="inline-flex items-center px-6 py-2.5 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-xl shadow-lg shadow-indigo-100 transition-all active:scale-95 text-sm">
                                Submit Review
                            </button>
                        </div>
                    </form>
                @else
                    <!-- (Existing content for recorded ratings remains unchanged) -->
                    <div class="p-6 sm:p-10 space-y-10">
                        <!-- ... -->
                        <!-- (Skipped content) -->
                    </div>
                @endif
            </div>
        @endif

        <!-- Order Items Section -->
        <div class="mt-12">
            <div class="flex items-center justify-between mb-6">
                <h2 class="text-xl font-bold text-gray-900 tracking-tight">Order Summary</h2>
                <span class="text-sm text-gray-500 font-medium">{{ $order->items->count() }} Items</span>
            </div>

            <div class="bg-white rounded-3xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-100">
                        <thead class="bg-gray-50/50">
                            <tr>
                                <th scope="col"
                                    class="px-4 py-3 text-left text-xs font-bold text-gray-400 uppercase tracking-widest">
                                    Product Details</th>
                                <th scope="col"
                                    class="px-4 py-3 text-left text-xs font-bold text-gray-400 uppercase tracking-widest">
                                    Price</th>
                                <th scope="col"
                                    class="px-4 py-3 text-center text-xs font-bold text-gray-400 uppercase tracking-widest">
                                    Qty</th>
                                <th scope="col"
                                    class="px-4 py-3 text-right text-xs font-bold text-gray-400 uppercase tracking-widest">
                                    Subtotal</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            @foreach($order->items as $item)
                                <tr class="group hover:bg-slate-50/50 transition-colors">
                                    <td class="px-4 py-4">
                                        <div class="flex items-center gap-4">
                                            <div
                                                class="h-16 w-16 flex-shrink-0 bg-gray-50 rounded-xl overflow-hidden border border-gray-100 shadow-sm group-hover:scale-105 transition-all duration-300">
                                                @if($item->variation && $item->variation->image_path)
                                                    <img class="h-full w-full object-cover" src="{{ $item->variation->image_path }}"
                                                        onerror="this.onerror=null; this.src='https://placehold.co/100x100?text=Lugani';"
                                                        alt="">
                                                @elseif($item->product->images->isNotEmpty())
                                                    <img class="h-full w-full object-cover"
                                                        src="{{ $item->product->images->first()->image_path }}"
                                                        onerror="this.onerror=null; this.src='https://placehold.co/100x100?text=Lugani';"
                                                        alt="">
                                                @else
                                                    <div
                                                        class="h-full w-full bg-gray-100 flex items-center justify-center text-[10px] font-black text-gray-400">
                                                        NO IMG</div>
                                                @endif
                                            </div>
                                            <div>
                                                <a href="{{ route('shop.product', $item->product->slug) }}"
                                                    class="text-sm font-bold text-gray-900 hover:text-indigo-600 transition-colors line-clamp-2">
                                                    {{ $item->product->name }}
                                                </a>
                                                @if(isset($item->options['attributes']))
                                                    <div class="mt-1 flex flex-wrap gap-2">
                                                        @foreach($item->options['attributes'] as $attr)
                                                            <span
                                                                class="inline-flex items-center px-2 py-0.5 rounded-md text-[10px] font-bold bg-slate-100 text-slate-600 border border-slate-200">
                                                                {{ $attr['attribute'] }}: {{ $attr['value'] }}
                                                            </span>
                                                        @endforeach
                                                    </div>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap text-sm font-medium text-gray-700">
                                        ₹{{ number_format($item->price, 2) }}</td>
                                    <td class="px-4 py-4 whitespace-nowrap text-center">
                                        <span
                                            class="inline-flex items-center justify-center w-6 h-6 rounded-full bg-slate-100 text-slate-700 text-xs font-bold">{{ $item->quantity }}</span>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap text-right text-sm font-bold text-gray-900">
                                        ₹{{ number_format($item->total, 2) }}</td>
                                </tr>
                            @endforeach
                        </tbody>
                        <tfoot class="bg-gray-50/80 border-t border-gray-100">
                            <tr>
                                <td colspan="3"
                                    class="px-4 py-3 text-right text-xs font-bold text-gray-500 uppercase tracking-wider">
                                    Subtotal</td>
                                <td class="px-4 py-3 text-right text-sm font-bold text-gray-900">
                                    ₹{{ number_format($order->subtotal, 2) }}</td>
                            </tr>
                            <tr>
                                <td colspan="3"
                                    class="px-4 py-3 text-right text-xs font-bold text-gray-500 uppercase tracking-wider">
                                    Shipping Fee</td>
                                <td class="px-4 py-3 text-right text-sm font-bold text-gray-900">
                                    ₹{{ number_format($order->shipping_cost, 2) }}</td>
                            </tr>
                            @if($order->discount > 0)
                                <tr>
                                    <td colspan="3"
                                        class="px-4 py-3 text-right text-xs font-bold text-green-600 uppercase tracking-wider">
                                        Promo Discount</td>
                                    <td class="px-4 py-3 text-right text-sm font-bold text-green-600">
                                        -₹{{ number_format($order->discount, 2) }}</td>
                                </tr>
                            @endif
                            @if($order->gift_card_discount > 0)
                                <tr>
                                    <td colspan="3"
                                        class="px-4 py-3 text-right text-xs font-bold text-green-600 uppercase tracking-wider">
                                        Gift Card Applied</td>
                                    <td class="px-4 py-3 text-right text-sm font-bold text-green-600">
                                        -₹{{ number_format($order->gift_card_discount, 2) }}</td>
                                </tr>
                            @endif
                            @if($order->points_discount > 0)
                                <tr>
                                    <td colspan="3"
                                        class="px-4 py-3 text-right text-xs font-bold text-green-600 uppercase tracking-wider">
                                        Points Spent ({{ $order->points_redeemed }})</td>
                                    <td class="px-4 py-3 text-right text-sm font-bold text-green-600">
                                        -₹{{ number_format($order->points_discount, 2) }}</td>
                                </tr>
                            @endif
                            <tr class="bg-indigo-600">
                                <td colspan="3"
                                    class="px-4 py-4 text-right text-sm font-bold text-white uppercase tracking-widest">
                                    Total Amount</td>
                                <td class="px-4 py-4 text-right text-xl font-black text-white">
                                    ₹{{ number_format($order->total, 2) }}</td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>

            <!-- Footer Actions -->
            <div class="mt-10 flex flex-col items-center gap-6">
                {{-- Cancellation --}}
                @if(in_array($order->status, ['pending', 'processing', 'processed']))
                    <div x-data="{ open: false }" class="relative">
                        <button @click="open = !open" type="button"
                            class="inline-flex items-center px-6 py-3 rounded-xl text-sm font-bold text-red-600 bg-red-50 hover:bg-red-100 border border-red-200 transition-all duration-300 shadow-sm hover:translate-y-px">
                            <svg class="h-4 w-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M6 18L18 6M6 6l12 12" />
                            </svg>
                            Need to cancel this order?
                        </button>

                        <div x-show="open" x-transition:enter="transition cubic-bezier(0.4, 0, 0.2, 1) duration-300"
                            x-transition:enter-start="opacity-0 translate-y-4 scale-95"
                            x-transition:enter-end="opacity-100 translate-y-0 scale-100" @click.away="open = false"
                            class="absolute bottom-full mb-4 left-1/2 -translate-x-1/2 w-[calc(100vw-2rem)] sm:w-[500px] max-w-3xl rounded-3xl shadow-2xl bg-white border border-gray-100 p-10 z-50">
                            <h4 class="text-xl font-black text-gray-900 mb-2">Cancel Order</h4>
                            <p class="text-sm text-gray-500 mb-6 font-medium leading-relaxed italic">Placing an order is
                                exciting, but we understand if you changed your mind.</p>

                            <form action="{{ route('user.orders.cancel', $order->id) }}" method="POST">
                                @csrf
                                <div class="mb-4">
                                    <label for="cancellation_reason"
                                        class="block text-[10px] font-black text-gray-400 mb-2 uppercase tracking-widest px-2">Reason</label>
                                    <select id="cancellation_reason" name="cancellation_reason" required
                                        class="block w-full border-2 border-gray-100 focus:border-red-500 focus:ring-red-500 rounded-2xl p-4 text-sm font-bold bg-slate-50 transition-all cursor-pointer">
                                        <option value="" disabled selected>Select a reason...</option>
                                        <option value="Found a better price">Found a better price</option>
                                        <option value="Order placed by mistake">Order placed by mistake</option>
                                        <option value="Delivery time is too long">Delivery time is too long</option>
                                        <option value="Changed my mind">Changed my mind</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                                <div class="flex flex-col gap-3">
                                    <button type="submit"
                                        class="w-full bg-red-600 hover:bg-red-700 text-white font-black py-4 rounded-2xl shadow-xl shadow-red-200 transition-all duration-300 active:scale-95"
                                        onclick="return confirm('Are you sure? This action is permanent.');">
                                        Confirm Cancellation
                                    </button>
                                    <button @click="open = false" type="button"
                                        class="w-full text-sm font-bold text-gray-400 py-2 hover:text-gray-600">Go Back</button>
                                </div>
                            </form>
                        </div>
                    </div>
                @endif

                {{-- Return Order --}}
                @if($order->status === 'completed' && !$order->returnRequest && $order->items->every(fn($item) => $item->product->is_returnable))
                    {{-- Optional: Check return window here visually or handled in controller --}}
                    <div x-data="{ open: false }" class="relative">
                        <button @click="open = !open" type="button"
                            class="inline-flex items-center px-8 py-4 rounded-2xl text-base font-black text-indigo-600 bg-indigo-50 hover:bg-indigo-100 border-2 border-indigo-200 transition-all duration-300 shadow-lg hover:shadow-indigo-100 hover:-translate-y-1">
                            <svg class="h-5 w-5 mr-3" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6" />
                            </svg>
                            Return Order
                        </button>

                        <div x-show="open" x-transition:enter="transition cubic-bezier(0.4, 0, 0.2, 1) duration-300"
                            x-transition:enter-start="opacity-0 translate-y-4 scale-95"
                            x-transition:enter-end="opacity-100 translate-y-0 scale-100" @click.away="open = false"
                            class="absolute bottom-full mb-4 left-1/2 -translate-x-1/2 w-[calc(100vw-2rem)] sm:w-[500px] max-w-3xl rounded-3xl shadow-2xl bg-white border border-gray-100 p-10 z-50">
                            <h4 class="text-xl font-black text-gray-900 mb-2">Request Return</h4>
                            <p class="text-sm text-gray-500 mb-6 font-medium leading-relaxed italic">Detailed information helps
                                us process your return faster.</p>

                            <form action="{{ route('user.orders.return', $order->id) }}" method="POST">
                                @csrf
                                <div class="mb-4">
                                    <label for="return_reason"
                                        class="block text-[10px] font-black text-gray-400 mb-2 uppercase tracking-widest px-2">Reason
                                        for Return</label>
                                    <textarea id="return_reason" name="reason" rows="4" required minlength="10"
                                        placeholder="Please describe why you are returning this order..."
                                        class="block w-full border-2 border-gray-100 focus:border-indigo-500 focus:ring-indigo-500 rounded-2xl p-4 text-sm font-bold bg-slate-50 transition-all"></textarea>
                                </div>
                                <div class="flex flex-col gap-3">
                                    <button type="submit"
                                        class="w-full bg-indigo-600 hover:bg-indigo-700 text-white font-black py-4 rounded-2xl shadow-xl shadow-indigo-200 transition-all duration-300 active:scale-95">
                                        Submit Return Request
                                    </button>
                                    <button @click="open = false" type="button"
                                        class="w-full text-sm font-bold text-gray-400 py-2 hover:text-gray-600">Cancel</button>
                                </div>
                            </form>
                        </div>
                    </div>
                @endif

                @if($order->returnRequest)
                    <div class="bg-indigo-50 rounded-2xl p-6 border border-indigo-100 text-center w-full max-w-2xl">
                        <h3 class="text-lg font-black text-indigo-900">Return Request Status:
                            {{ ucfirst(str_replace('_', ' ', $order->returnRequest->status)) }}
                        </h3>
                        <p class="text-indigo-700 mt-2 font-medium">
                            {{ $order->returnRequest->admin_note ?? 'Your request is being processed.' }}
                        </p>
                        @if($order->returnRequest->refund_amount)
                            <p class="text-green-600 mt-2 font-black">Refund Amount: ₹{{ $order->returnRequest->refund_amount }}</p>
                        @endif

                        @if($order->returnRequest->refund_proof_path)
                            <div class="mt-4 pt-4 border-t border-indigo-200">
                                <a href="{{ asset('storage/' . $order->returnRequest->refund_proof_path) }}" target="_blank"
                                    class="inline-flex items-center text-sm font-bold text-indigo-700 hover:text-indigo-900 underline">
                                    <svg class="h-4 w-4 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                                    </svg>
                                    View Refund Proof
                                </a>
                            </div>
                        @endif
                    </div>
                @endif
            </div>
        </div>
    </div>
@endsection