<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>@yield('title', $settings->site_name ?? 'Hair Niche')</title>
    <meta name="description" content="@yield('meta_description', '')">
    <meta name="keywords" content="@yield('meta_keywords', '')">
    @vite(['resources/css/app.css', 'resources/js/app.js'])
    <link href="https://fonts.googleapis.com/css2?family=Urbanist:ital,wght@0,100..900;1,100..900&display=swap"
        rel="stylesheet">
    @if(optional($settings)->favicon)
        <link rel="icon" href="{{ Storage::url($settings->favicon) }}" type="image/x-icon">
    @endif
    <script defer src="https://cdn.jsdelivr.net/npm/@alpinejs/collapse@3.x.x/dist/cdn.min.js"></script>
    <script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <style>
        body {
            font-family: 'Urbanist', sans-serif;
            font-size: 0.9375rem;
            /* 15px */
            line-height: 1.6;
            letter-spacing: 0.01em;
        }

        [x-cloak] {
            display: none !important;
        }

        h1,
        h2,
        h3,
        h4,
        h5,
        h6 {
            letter-spacing: -0.01em;
        }
    </style>
    @stack('meta')
    @stack('schema_markup')
    @stack('head_scripts')
</head>

<body class="bg-white text-gray-900 antialiased flex flex-col min-h-screen">

    <!-- Announcement Bar -->
    @if($settings && $settings->free_shipping_enabled)
        <div class="bg-gray-900 text-white text-xs font-medium text-center py-2 tracking-wide uppercase">
            Free shipping on all orders over ₹{{ number_format($settings->free_shipping_amount, 0) }}
        </div>
    @endif

    <!-- Top contact bar for Desktop -->
    <div
        class="hidden md:flex justify-between items-center bg-gray-50 border-b border-gray-100 px-4 py-1 sm:px-6 lg:px-8 text-xs text-gray-500">
        <div class="flex items-center space-x-6">
            @if(optional($settings)->phone_support)
                <a href="tel:{{ $settings->phone_support }}"
                    class="flex items-center hover:text-gray-900 transition-colors">
                    <svg class="h-3 w-3 mr-1.5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                    </svg>
                    {{ $settings->phone_support }}
                </a>
            @endif
            @if(optional($settings)->email_support)
                <a href="mailto:{{ $settings->email_support }}"
                    class="flex items-center hover:text-gray-900 transition-colors">
                    <svg class="h-3 w-3 mr-1.5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                    </svg>
                    {{ $settings->email_support }}
                </a>
            @endif
        </div>
        <div class="flex items-center space-x-4">
            <a href="{{ route('pages.contact') }}" class="hover:text-gray-900">Contact Us</a>
            <span class="text-gray-300">|</span>
            @if(Auth::check())
                <a href="{{ route('user.dashboard') }}" class="hover:text-gray-900">My Dashboard</a>
                <span class="text-gray-300">|</span>
                <form method="POST" action="{{ route('logout') }}" class="inline">
                    @csrf
                    <button type="submit" class="hover:text-gray-900">Logout</button>
                </form>
            @else
                <a href="{{ route('login') }}" class="hover:text-gray-900">Login</a>
                <span class="text-gray-300">|</span>
                <a href="{{ route('register') }}" class="hover:text-gray-900">Sign Up</a>
            @endif
        </div>
    </div>

    <!-- Header -->
    <header class="sticky top-0 z-50 bg-white/80 backdrop-blur-md border-b border-gray-100"
        x-data="{ mobileMenu: false }">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-20">
                <!-- Mobile Menu Button -->
                <div class="flex items-center md:hidden">
                    <button @click="mobileMenu = !mobileMenu" type="button"
                        class="text-gray-500 hover:text-gray-900 p-2">
                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M4 6h16M4 12h16M4 18h16" />
                        </svg>
                    </button>
                </div>

                <!-- Logo -->
                <div class="flex-shrink-0 flex items-center">
                    <a href="{{ route('home') }}" class="text-2xl font-bold tracking-tight text-gray-900 uppercase">
                        @if(optional($settings)->site_logo)
                            <img src="{{ Storage::url($settings->site_logo) }}" alt="{{ $settings->site_name }}"
                                class="w-[120px] h-[80px] object-contain">
                        @else
                            {{ optional($settings)->site_name ?? 'LUGANI' }}
                        @endif
                    </a>
                </div>

                <!-- Desktop Nav -->
                <nav class="hidden md:flex space-x-8 items-center h-full">
                    <a href="{{ route('home') }}"
                        class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide flex items-center h-full">Home</a>

                    <!-- Mega Menu for Categories -->
                    <div class="group h-full flex items-center">
                        <a href="{{ route('shop.index') }}"
                            class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide cursor-pointer flex items-center h-full border-b-2 border-transparent group-hover:border-gray-900 transition-colors">
                            Categories
                        </a>
                        <!-- Mega Menu Dropdown -->
                        <div
                            class="absolute left-0 top-20 w-full bg-white border-t border-gray-100 shadow-xl opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300 transform translate-y-2 group-hover:translate-y-0 z-50">
                            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                                <div class="grid grid-cols-4 gap-x-0 gap-y-10">
                                    @if(isset($menuCategories))
                                        @foreach($menuCategories as $category)
                                            <div
                                                class="pr-8 {{ $loop->index % 4 == 0 ? 'pl-0' : 'pl-8 border-l border-gray-200' }}">
                                                <a href="{{ route('shop.category', $category->slug) }}"
                                                    class="block group/item mb-4" {{ $category->is_nofollow ? 'rel=nofollow' : '' }}>
                                                    <span
                                                        class="block font-bold text-base text-gray-800 uppercase tracking-widest hover:text-indigo-600 transition-colors flex items-center">
                                                        {{ $category->name }}
                                                    </span>
                                                </a>
                                                <ul class="space-y-2.5">
                                                    @foreach($category->children as $child)
                                                        <li>
                                                            <a href="{{ route('shop.subcategory', ['category' => $category->slug, 'subcategory' => $child->slug]) }}"
                                                                class="flex items-center text-sm font-medium text-gray-500 hover:text-indigo-600 hover:translate-x-1 transition-all duration-200" {{ $child->is_nofollow ? 'rel=nofollow' : '' }}>
                                                                <span
                                                                    class="w-1.5 h-1.5 rounded-full bg-gray-200 mr-2 group-hover:bg-indigo-400 transition-colors"></span>
                                                                {{ $child->name }}
                                                            </a>
                                                        </li>
                                                    @endforeach
                                                </ul>
                                            </div>
                                        @endforeach
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>

                    <a href="{{ route('shop.index') }}"
                        class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide flex items-center h-full">Shop
                        All</a>
                    <a href="{{ route('blogs.index') }}"
                        class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide flex items-center h-full">Blog</a>
                    <a href="{{ route('pages.about') }}"
                        class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide flex items-center h-full">About</a>
                    <a href="{{ route('pages.contact') }}"
                        class="text-sm font-medium text-gray-700 hover:text-gray-900 uppercase tracking-wide flex items-center h-full">Contact</a>
                </nav>

                <!-- Search Bar -->
                <div class="hidden md:flex items-center flex-1 max-w-sm mx-6" x-data="{
                         searchQuery: '',
                         suggestions: [],
                         fetchSuggestions() {
                             if(this.searchQuery.length < 2) {
                                 this.suggestions = [];
                                 return;
                             }
                             fetch('{{ route('shop.search.suggestions') }}?query=' + this.searchQuery)
                                 .then(res => res.json())
                                 .then(data => {
                                     this.suggestions = data;
                                 });
                         },
                         performSearch() {
                             window.location.href = '{{ route('shop.index') }}?search=' + this.searchQuery;
                         }
                     }">
                    <div class="relative w-full">
                        <input type="text" x-model="searchQuery" @input.debounce.300ms="fetchSuggestions"
                            @keydown.enter="performSearch" placeholder="Search products..."
                            class="w-full bg-gray-50 border border-gray-200 rounded-full py-2 pl-4 pr-10 text-sm focus:ring-1 focus:ring-indigo-500 focus:border-indigo-500 focus:bg-white transition-all duration-200">
                        <button @click="performSearch"
                            class="absolute right-0 top-0 h-full px-3 text-gray-400 hover:text-indigo-600 transition-colors">
                            <svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                            </svg>
                        </button>

                        <!-- Live Search Results -->
                        <div x-show="suggestions.length > 0" @click.away="suggestions = []"
                            x-transition:enter="transition ease-out duration-100"
                            x-transition:enter-start="transform opacity-0 scale-95"
                            x-transition:enter-end="transform opacity-100 scale-100"
                            x-transition:leave="transition ease-in duration-75"
                            x-transition:leave-start="transform opacity-100 scale-100"
                            x-transition:leave-end="transform opacity-0 scale-95"
                            class="absolute top-full left-0 right-0 mt-2 bg-white rounded-xl shadow-2xl border border-gray-100 py-2 z-50 overflow-hidden ring-1 ring-black ring-opacity-5">

                            <template x-for="product in suggestions" :key="product.url">
                                <a :href="product.url"
                                    class="group flex items-center px-4 py-3 hover:bg-indigo-50 transition-colors border-b border-gray-50 last:border-0">
                                    <div class="flex-shrink-0 relative">
                                        <img :src="product.image"
                                            class="h-12 w-12 object-cover rounded-lg border border-gray-200 group-hover:border-indigo-200"
                                            alt="">
                                    </div>
                                    <div class="ml-4 flex-1">
                                        <!-- Product/Variation Name -->
                                        <div class="text-sm font-medium text-gray-900 group-hover:text-indigo-700 transition-colors"
                                            x-text="product.name"></div>
                                        <!-- Price -->
                                        <div class="text-xs font-bold text-indigo-600 mt-0.5"
                                            x-text="'₹' + Number(product.price).toLocaleString()">
                                        </div>
                                    </div>
                                    <div class="ml-auto">
                                        <svg class="h-5 w-5 text-gray-400 group-hover:text-indigo-500 opacity-0 group-hover:opacity-100 transition-all transform group-hover:translate-x-1"
                                            xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
                                            stroke="currentColor">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M9 5l7 7-7 7" />
                                        </svg>
                                    </div>
                                </a>
                            </template>

                            <a @click="performSearch"
                                class="block px-4 py-3 text-xs font-bold text-center text-gray-500 hover:bg-gray-50 hover:text-indigo-600 transition-colors cursor-pointer uppercase tracking-widest border-t border-gray-100">
                                See all results for "<span x-text="searchQuery"></span>"
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Icons -->
                <div class="flex items-center space-x-6">
                    <!-- Only show auth user icon on desktop if you want, or keep logic simple -->
                    @if(Auth::check())
                        @if(Auth::user()->hasRole('delivery_agent'))
                            <a href="{{ route('delivery.notifications') }}"
                                class="text-blue-500 hover:text-blue-700 transition-colors relative" title="Notifications">
                        @else
                                <a href="{{ route('user.tickets.index') }}"
                                    class="text-blue-500 hover:text-blue-700 transition-colors relative" title="Notifications">
                            @endif
                                <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                        d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" />
                                </svg>
                                @php
                                    $unreadCount = Auth::user()->unreadNotifications->count();
                                 @endphp
                                @if($unreadCount > 0)
                                    <span
                                        class="absolute -top-1 -right-1 bg-red-600 text-white text-[10px] w-4 h-4 flex items-center justify-center rounded-full border border-white">
                                        {{ $unreadCount }}
                                    </span>
                                @endif
                            </a>
                            @if(Auth::user()->hasRole('delivery_agent'))
                                <a href="{{ route('delivery.dashboard') }}"
                                    class="text-indigo-600 hover:text-indigo-800 transition-colors" title="My Account">
                            @else
                                    <a href="{{ route('user.dashboard') }}"
                                        class="text-indigo-600 hover:text-indigo-800 transition-colors" title="My Account">
                                @endif
                                    <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                            d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                    </svg>
                                </a>
                    @else
                                <a href="{{ route('login') }}"
                                    class="text-indigo-600 hover:text-indigo-800 transition-colors" title="Login">
                                    <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                            d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                                    </svg>
                                </a>
                            @endif

                            <a href="{{ route('user.wishlist') }}"
                                class="text-rose-500 hover:text-rose-700 transition-colors relative" title="Wishlist">
                                <svg class="h-6 w-6" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"
                                    stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                        d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                </svg>
                                @php
                                    $wishlistCount = Auth::check() ? \App\Models\Wishlist::where('user_id', Auth::id())->count() : 0;
                                @endphp
                                <span id="wishlist-count-badge"
                                    class="{{ $wishlistCount > 0 ? '' : 'hidden' }} absolute -top-1 -right-1 bg-red-600 text-white text-[10px] w-4 h-4 flex items-center justify-center rounded-full border border-white">
                                    {{ $wishlistCount }}
                                </span>
                            </a>

                            <a href="{{ route('cart.index') }}"
                                class="text-amber-600 hover:text-amber-800 transition-colors relative group">
                                <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                        d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                                </svg>
                                <!-- Badge logic would ideally go here -->
                                @php
                                    $cartCount = 0;
                                    if (Auth::check()) {
                                        $cart = \App\Models\Cart::where('user_id', Auth::id())->first();
                                        if ($cart)
                                            $cartCount = $cart->items->sum('quantity');
                                    } else {
                                        $sessId = session('cart_session_id');
                                        if ($sessId) {
                                            $cart = \App\Models\Cart::where('session_id', $sessId)->first();
                                            if ($cart)
                                                $cartCount = $cart->items->sum('quantity');
                                        }
                                    }
                                @endphp
                                @if($cartCount > 0)
                                    <span
                                        class="absolute -top-1 -right-1 bg-black text-white text-[10px] w-4 h-4 flex items-center justify-center rounded-full">{{ $cartCount }}</span>
                                @endif
                            </a>
                </div>
            </div>
        </div>

        <!-- Mobile Menu -->
        <div x-show="mobileMenu" x-cloak 
            x-transition:enter="transition ease-out duration-200"
            x-transition:enter-start="opacity-0 transform -translate-y-2"
            x-transition:enter-end="opacity-100 transform translate-y-0"
            x-transition:leave="transition ease-in duration-150"
            x-transition:leave-start="opacity-100 transform translate-y-0"
            x-transition:leave-end="opacity-0 transform -translate-y-2"
            class="md:hidden border-t border-gray-100 bg-white shadow-lg max-h-[80vh] overflow-y-auto">
            <div class="pt-2 pb-4 space-y-1 px-4">
                <!-- Home -->
                <a href="{{ route('home') }}"
                    class="block py-3 text-base font-medium text-gray-900 border-b border-gray-50 hover:bg-gray-50 transition-colors">
                    Home
                </a>

                <!-- Categories with Mega Menu -->
                <div x-data="{ categoriesOpen: false }" class="border-b border-gray-50">
                    <button @click="categoriesOpen = !categoriesOpen"
                        class="w-full flex justify-between items-center py-3 text-base font-medium text-gray-900 hover:bg-gray-50 transition-colors">
                        <span>Categories</span>
                        <svg class="h-5 w-5 transform transition-transform" :class="{ 'rotate-180': categoriesOpen }"
                            fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7" />
                        </svg>
                    </button>

                    <!-- Categories Dropdown -->
                    <div x-show="categoriesOpen" x-collapse class="pl-4 pb-2 space-y-2">
                        @if(isset($menuCategories))
                            @foreach($menuCategories as $category)
                                <div x-data="{ subOpen: false }" class="border-l-2 border-gray-200 pl-3">
                                    <!-- Parent Category -->
                                    <div class="flex items-center justify-between">
                                        <a href="{{ route('shop.category', $category->slug) }}"
                                            class="flex-1 py-2 text-sm font-semibold text-gray-800 hover:text-[#9B2D69] transition-colors"
                                            {{ $category->is_nofollow ? 'rel=nofollow' : '' }}>
                                            {{ $category->name }}
                                        </a>
                                        @if($category->children->count() > 0)
                                            <button @click="subOpen = !subOpen" class="p-2">
                                                <svg class="h-4 w-4 transform transition-transform" :class="{ 'rotate-180': subOpen }"
                                                    fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7" />
                                                </svg>
                                            </button>
                                        @endif
                                    </div>

                                    <!-- Subcategories -->
                                    @if($category->children->count() > 0)
                                        <div x-show="subOpen" x-collapse class="pl-3 space-y-1 mt-1">
                                            @foreach($category->children as $child)
                                                <a href="{{ route('shop.subcategory', ['category' => $category->slug, 'subcategory' => $child->slug]) }}"
                                                    class="flex items-center py-1.5 text-sm text-gray-600 hover:text-[#9B2D69] transition-colors"
                                                    {{ $child->is_nofollow ? 'rel=nofollow' : '' }}>
                                                    <span class="w-1.5 h-1.5 rounded-full bg-gray-300 mr-2"></span>
                                                    {{ $child->name }}
                                                </a>
                                            @endforeach
                                        </div>
                                    @endif
                                </div>
                            @endforeach
                        @endif
                    </div>
                </div>

                <!-- Shop All -->
                <a href="{{ route('shop.index') }}"
                    class="block py-3 text-base font-medium text-gray-900 border-b border-gray-50 hover:bg-gray-50 transition-colors">
                    Shop All
                </a>

                <!-- Blog -->
                <a href="{{ route('blogs.index') }}"
                    class="block py-3 text-base font-medium text-gray-900 border-b border-gray-50 hover:bg-gray-50 transition-colors">
                    Blog
                </a>

                <!-- About -->
                <a href="{{ route('pages.about') }}"
                    class="block py-3 text-base font-medium text-gray-900 border-b border-gray-50 hover:bg-gray-50 transition-colors">
                    About
                </a>

                <!-- Contact -->
                <a href="{{ route('pages.contact') }}"
                    class="block py-3 text-base font-medium text-gray-900 border-b border-gray-50 hover:bg-gray-50 transition-colors">
                    Contact
                </a>

                <!-- User Account Section -->
                @if(Auth::check())
                    <a href="{{ route('user.dashboard') }}"
                        class="block py-3 text-base font-medium text-[#9B2D69] border-t border-gray-200 mt-2 hover:bg-gray-50 transition-colors">
                        My Dashboard
                    </a>
                    <form method="POST" action="{{ route('logout') }}">
                        @csrf
                        <button type="submit"
                            class="block w-full text-left py-3 text-base font-medium text-red-600 hover:bg-gray-50 transition-colors">
                            Logout
                        </button>
                    </form>
                @else
                    <div class="flex space-x-4 border-t border-gray-200 pt-4 mt-2">
                        <a href="{{ route('login') }}" 
                            class="flex-1 text-center py-2 px-4 bg-[#9B2D69] text-white rounded-lg font-medium hover:bg-[#7a2454] transition-colors">
                            Login
                        </a>
                        <a href="{{ route('register') }}" 
                            class="flex-1 text-center py-2 px-4 border-2 border-[#9B2D69] text-[#9B2D69] rounded-lg font-medium hover:bg-[#9B2D69] hover:text-white transition-colors">
                            Sign Up
                        </a>
                    </div>
                @endif
            </div>
        </div>
    </header>

    <!-- Main Content -->
    <main class="flex-grow">
        @include('components.custom-offers-container')
        @yield('content')
    </main>

    <!-- Footer -->
    <footer class="bg-gray-900 text-white pt-16 pb-8">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-12 mb-12">
                <div class="space-y-4">
                    <h4 class="text-lg font-bold uppercase tracking-wider">About</h4>
                    <p class="text-gray-400 text-sm leading-relaxed">
                        @if(optional($settings)->address_head)
                            {{ $settings->address_head }}<br>
                        @endif
                        Elevating your lifestyle with premium quality products. Designed for the modern aesthetic.
                    </p>
                </div>
                <div>
                    <h4 class="text-sm font-bold uppercase tracking-wider mb-4">Quick Links</h4>
                    <ul class="space-y-2 text-sm text-gray-400">
                        <li><a href="{{ route('home') }}" class="hover:text-white transition-colors">Home</a></li>
                        <li><a href="{{ route('shop.index') }}" class="hover:text-white transition-colors">Shop</a></li>
                        <li><a href="{{ route('user.support') }}" class="hover:text-white transition-colors">Help &
                                Support</a></li>
                        <li><a href="{{ route('login') }}" class="hover:text-white transition-colors">Login</a></li>
                        <li><a href="{{ route('register') }}" class="hover:text-white transition-colors">Register</a>
                        </li>
                    </ul>
                </div>
                <div>
                    <h4 class="text-sm font-bold uppercase tracking-wider mb-4">Useful Links</h4>
                    @php
                        $footerPages = \App\Models\Page::whereIn('slug', [
                            'about-us', 
                            'cancellation-policy', 
                            'data-deletion', 
                            'privacy-policy', 
                            'pricing-refund-policy', 
                            'shipping-policy', 
                            'terms-conditions'
                        ])->get()->keyBy('slug');
                    @endphp
                    <ul class="space-y-2 text-sm text-gray-400">
                        <li>
                            <a href="{{ route('pages.about') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['about-us']) && $footerPages['about-us']->is_nofollow) rel="nofollow" @endif>
                                About us
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.cancellation-policy') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['cancellation-policy']) && $footerPages['cancellation-policy']->is_nofollow) rel="nofollow" @endif>
                                Cancellation Policy
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.data-deletion') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['data-deletion']) && $footerPages['data-deletion']->is_nofollow) rel="nofollow" @endif>
                                Data Deletion
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.privacy-policy') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['privacy-policy']) && $footerPages['privacy-policy']->is_nofollow) rel="nofollow" @endif>
                                Privacy Policy
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.pricing-refund-policy') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['pricing-refund-policy']) && $footerPages['pricing-refund-policy']->is_nofollow) rel="nofollow" @endif>
                                Pricing & Refund Policy
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.shipping-policy') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['shipping-policy']) && $footerPages['shipping-policy']->is_nofollow) rel="nofollow" @endif>
                                Shipping Policy
                            </a>
                        </li>
                        <li>
                            <a href="{{ route('pages.terms-conditions') }}" class="hover:text-white transition-colors"
                               @if(isset($footerPages['terms-conditions']) && $footerPages['terms-conditions']->is_nofollow) rel="nofollow" @endif>
                                Terms & Conditions
                            </a>
                        </li>
                        <li><a href="{{ route('reseller.create') }}" class="hover:text-white transition-colors">Become a Reseller</a></li>
                    </ul>
                </div>
                <div>
                    <h4 class="text-sm font-bold uppercase tracking-wider mb-4">Stay in the Loop</h4>
                    <form action="{{ route('newsletter.subscribe') }}" method="POST" class="flex flex-col space-y-2">
                        @csrf
                        <input type="email" name="email" placeholder="Enter your email" required
                            class="bg-gray-800 border border-gray-700 text-white px-4 py-2 rounded focus:outline-none focus:border-white text-sm">
                        <button type="submit"
                            class="bg-white text-black px-4 py-2 rounded font-medium text-sm hover:bg-gray-200 transition-colors uppercase tracking-wide">Subscribe</button>
                    </form>
                    @if(session('success'))
                        <p class="text-green-500 text-xs mt-2">{{ session('success') }}</p>
                    @endif
                    @error('email')
                        <p class="text-red-500 text-xs mt-2">{{ $message }}</p>
                    @enderror
                    @if($settings)
                        <div class="mt-4 text-xs text-gray-500">
                            @if($settings->email_support)
                                <p>Support: <a href="mailto:{{ $settings->email_support }}"
                                        class="hover:text-white">{{ $settings->email_support }}</a></p>
                            @endif
                            @if($settings->phone_support)
                                <p>Phone: <a href="tel:{{ $settings->phone_support }}"
                                        class="hover:text-white">{{ $settings->phone_support }}</a></p>
                            @endif
                        </div>
                    @endif
                    <div class="flex space-x-4 pt-2">
                        @if(optional($settings)->social_links && is_array($settings->social_links))
                            @foreach($settings->social_links as $platform => $linkData)
                                @php
                                    // Handle backward compatibility (string) vs new format (array)
                                    $url = is_array($linkData) ? ($linkData['url'] ?? '#') : $linkData;
                                    $isNewWindow = is_array($linkData) && ($linkData['new_window'] ?? false);
                                    $isNofollow = is_array($linkData) && ($linkData['nofollow'] ?? false);

                                    $target = $isNewWindow ? '_blank' : '_self';
                                    $relArray = [];
                                    if ($isNofollow)
                                        $relArray[] = 'nofollow';
                                    
                                    $rel = implode(' ', $relArray);
                                @endphp
                                @if($url && $url !== '#')
                                    <a href="{{ $url }}" target="{{ $target }}" @if($rel) rel="{{ $rel }}" @endif
                                        class="text-gray-400 hover:text-white transition-colors">
                                        <span class="sr-only">{{ ucfirst($platform) }}</span>
                                        @if(str_contains(strtolower($platform), 'facebook'))
                                            <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
                                                <path
                                                    d="M22 12c0-5.523-4.477-10-10-10S2 6.477 2 12c0 4.991 3.657 9.128 8.438 9.878v-6.987h-2.54V12h2.54V9.797c0-2.506 1.492-3.89 3.777-3.89 1.094 0 2.238.195 2.238.195v2.46h-1.26c-1.243 0-1.63.771-1.63 1.562V12h2.773l-.443 2.89h-2.33v6.988C18.343 21.128 22 16.991 22 12z" />
                                            </svg>
                                        @elseif(str_contains(strtolower($platform), 'twitter') || str_contains(strtolower($platform), 'x'))
                                            <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
                                                <path
                                                    d="M8.29 20.251c7.547 0 11.675-6.253 11.675-11.675 0-.178 0-.355-.012-.53A8.348 8.348 0 0022 5.92a8.19 8.19 0 01-2.357.646 4.118 4.118 0 001.804-2.27 8.224 8.224 0 01-2.605.996 4.107 4.107 0 00-6.993 3.743 11.65 11.65 0 01-8.457-4.287 4.106 4.106 0 001.27 5.477A4.072 4.072 0 012.8 9.713v.052a4.105 4.105 0 003.292 4.022 4.095 4.095 0 01-1.853.07 4.108 4.108 0 003.834 2.85A8.233 8.233 0 012 18.407a11.616 11.616 0 006.29 1.84" />
                                            </svg>
                                        @elseif(str_contains(strtolower($platform), 'instagram'))
                                            <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
                                                <path fill-rule="evenodd"
                                                    d="M12.315 2c2.43 0 2.784.013 3.808.06 1.064.049 1.791.218 2.427.465a4.902 4.902 0 011.772 1.153 4.902 4.902 0 011.153 1.772c.247.636.416 1.363.465 2.427.048 1.067.06 1.407.06 4.123v.08c0 2.643-.012 2.987-.06 4.043-.049 1.064-.218 1.791-.465 2.427a4.902 4.902 0 01-1.153 1.772 4.902 4.902 0 01-1.772 1.153c-.636.247-1.363.416-2.427.465-1.067.048-1.407.06-4.123.06h-.08c-2.643 0-2.987-.012-4.043-.06-1.064-.049-1.791-.218-2.427-.465a4.902 4.902 0 01-1.772-1.153 4.902 4.902 0 01-1.153-1.772c-.247-.636-.416-1.363-.465-2.427-.047-1.024-.06-1.379-.06-3.808v-.63c0-2.43.013-2.784.06-3.808.049-1.064.218-1.791.465-2.427a4.902 4.902 0 011.153-1.772A4.902 4.902 0 015.468 2.373c.636-.247 1.363-.416 2.427-.465C8.901 2.013 9.256 2 11.685 2h.63zm-.081 1.802h-.468c-2.456 0-2.784.011-3.807.058-.975.045-1.504.207-1.857.344-.467.182-.8.398-1.15.748-.35.35-.566.683-.748 1.15-.137.353-.3.882-.344 1.857-.047 1.023-.058 1.351-.058 3.807v.468c0 2.456.011 2.784.058 3.807.045.975.207 1.504.344 1.857.182.466.399.8.748 1.15.35.35.683.566 1.15.748.353.137.882.3 1.857.344 1.054.048 1.37.058 4.041.058h.08c2.597 0 2.917-.01 3.96-.058.976-.045 1.505-.207 1.858-.344.466-.182.8-.398 1.15-.748.35-.35.566-.683.748-1.15.137-.353.3-.882.344-1.857.048-1.055.058-1.37.058-4.041v-.08c0-2.597-.01-2.917-.058-3.96-.045-.976-.207-1.505-.344-1.858a3.097 3.097 0 00-.748-1.15 3.098 3.098 0 00-1.15-.748c-.353-.137-.882-.3-1.857-.344-1.023-.047-1.351-.058-3.807-.058zM12 6.865a5.135 5.135 0 110 10.27 5.135 5.135 0 010-10.27zm0 1.802a3.333 3.333 0 100 6.666 3.333 3.333 0 000-6.666zm5.338-3.205a1.2 1.2 0 110 2.4 1.2 1.2 0 010-2.4z"
                                                    clip-rule="evenodd" />
                                            </svg>
                                        @elseif(str_contains(strtolower($platform), 'youtube'))
                                            <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24">
                                                <path
                                                    d="M21.582 5.184c-.233-.873-.913-1.553-1.786-1.786C18.22 3 12 3 12 3s-6.22 0-7.796.398c-.873.233-1.553.913-1.786 1.786C2 6.76 2 12 2 12s0 5.24.398 7.618c.233.873.913 1.553 1.786 1.786 1.576.398 7.796.398 7.796.398s6.22 0 7.796-.398c.873-.233 1.553-.913 1.786-1.786C22 17.24 22 12 22 12s0-5.24-.398-7.618zM9.545 15.568V8.432L15.818 12l-6.273 3.568z" />
                                            </svg>
                                        @elseif(str_contains(strtolower($platform), 'pinterest'))
                                            <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                                <path d="M12.017 0C5.396 0 .029 5.367.029 11.987c0 5.079 3.158 9.417 7.618 11.162-.105-.949-.199-2.403.041-3.439.219-.937 1.406-5.957 1.406-5.957s-.359-.72-.359-1.781c0-1.663.967-2.911 2.168-2.911 1.024 0 1.518.769 1.518 1.688 0 1.029-.653 2.567-.992 3.992-.285 1.193.6 2.165 1.775 2.165 2.128 0 3.768-2.245 3.768-5.487 0-2.861-2.063-4.869-5.008-4.869-3.41 0-5.409 2.562-5.409 5.199 0 1.033.394 2.143.889 2.741.099.12.112.225.085.345-.09.375-.293 1.199-.334 1.363-.053.225-.172.271-.399.165-1.495-.69-2.433-2.878-2.433-4.646 0-3.776 2.748-7.252 7.951-7.252 4.173 0 7.41 2.967 7.41 6.923 0 4.135-2.607 7.462-6.233 7.462-1.214 0-2.354-.629-2.758-1.379l-.749 2.848c-.269 1.045-1.004 2.352-1.498 3.146 1.123.345 2.306.535 3.55.535 6.607 0 11.985-5.365 11.985-11.987C23.97 5.367 18.62 0 12.017 0z"/>
                                            </svg>
                                        @else
                                            <!-- Default Link Icon -->
                                            <svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1" />
                                            </svg>
                                        @endif
                                    </a>
                                @endif
                            @endforeach
                        @endif
                    </div>
                </div>
            </div>

            <!-- App Download Section -->
            <div class="border-t border-gray-800 pt-8 pb-6">
                <div class="text-center mb-6">
                    <h3 class="text-lg font-bold text-white mb-2">Download Our App</h3>
                    <p class="text-gray-400 text-sm">Get exclusive deals and faster checkout</p>
                </div>
                <div class="flex flex-col sm:flex-row justify-center items-center gap-4">
                    <!-- Google Play Store -->
                    <a href="#" class="inline-flex items-center px-6 py-3 bg-black hover:bg-gray-900 text-white rounded-lg transition-all duration-200 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5">
                        <svg class="w-8 h-8 mr-3" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M3,20.5V3.5C3,2.91 3.34,2.39 3.84,2.15L13.69,12L3.84,21.85C3.34,21.6 3,21.09 3,20.5M16.81,15.12L6.05,21.34L14.54,12.85L16.81,15.12M20.16,10.81C20.5,11.08 20.75,11.5 20.75,12C20.75,12.5 20.5,12.92 20.16,13.19L17.89,14.5L15.39,12L17.89,9.5L20.16,10.81M6.05,2.66L16.81,8.88L14.54,11.15L6.05,2.66Z" />
                        </svg>
                        <div class="text-left">
                            <div class="text-xs">GET IT ON</div>
                            <div class="text-base font-semibold -mt-1">Google Play</div>
                        </div>
                    </a>

                    <!-- Apple App Store -->
                    <a href="#" class="inline-flex items-center px-6 py-3 bg-black hover:bg-gray-900 text-white rounded-lg transition-all duration-200 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5">
                        <svg class="w-8 h-8 mr-3" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M18.71,19.5C17.88,20.74 17,21.95 15.66,21.97C14.32,22 13.89,21.18 12.37,21.18C10.84,21.18 10.37,21.95 9.1,22C7.79,22.05 6.8,20.68 5.96,19.47C4.25,17 2.94,12.45 4.7,9.39C5.57,7.87 7.13,6.91 8.82,6.88C10.1,6.86 11.32,7.75 12.11,7.75C12.89,7.75 14.37,6.68 15.92,6.84C16.57,6.87 18.39,7.1 19.56,8.82C19.47,8.88 17.39,10.1 17.41,12.63C17.44,15.65 20.06,16.66 20.09,16.67C20.06,16.74 19.67,18.11 18.71,19.5M13,3.5C13.73,2.67 14.94,2.04 15.94,2C16.07,3.17 15.6,4.35 14.9,5.19C14.21,6.04 13.07,6.7 11.95,6.61C11.8,5.46 12.36,4.26 13,3.5Z" />
                        </svg>
                        <div class="text-left">
                            <div class="text-xs">Download on the</div>
                            <div class="text-base font-semibold -mt-1">App Store</div>
                        </div>
                    </a>
                </div>
            </div>

            <div class="border-t border-gray-800 pt-8 flex flex-col md:flex-row justify-between items-center">
                <div class="mb-4 md:mb-0">
                    <p class="text-gray-500 text-xs">&copy; {{ date('Y') }} {{ $settings->site_name ?? 'Hair Niche By Shelly' }}.
                        All rights reserved.</p>
                    <div class="flex flex-wrap gap-4 mt-2 text-xs text-gray-500">
                        <a href="{{ route('pages.privacy-policy') }}"
                            class="hover:text-white transition-colors">Privacy</a>
                        <a href="{{ route('pages.terms-conditions') }}"
                            class="hover:text-white transition-colors">Terms</a>
                        <a href="{{ route('pages.cancellation-policy') }}"
                            class="hover:text-white transition-colors">Cancellation</a>
                        <a href="{{ route('reseller.create') }}" class="hover:text-white transition-colors">Become a
                            Reseller</a>
                        <a href="{{ route('reseller.login') }}"
                            class="hover:text-white transition-colors border-l border-gray-700 pl-4">Reseller Login</a>
                    </div>
                </div>
                <div class="flex space-x-4">
                    <!-- Payment Icons -->
                    <svg class="h-8 w-10 text-gray-500" viewBox="0 0 38 24" fill="currentColor">
                        <path d="M35 0H3C1.3 0 0 1.3 0 3v18c0 1.7 1.4 3 3 3h32c1.7 0 3-1.3 3-3V3c0-1.7-1.4-3-3-3z"
                            opacity=".1" />
                        <path
                            d="M35 1c1.1 0 2 .9 2 2v18c0 1.1-.9 2-2 2H3c-1.1 0-2-.9-2-2V3c0-1.1.9-2 2-2h32m0-1H3C1.3 0 0 1.3 0 3v18c0 1.7 1.4 3 3 3h32c1.7 0 3-1.3 3-3V3c0-1.7-1.4-3-3-3z"
                            fill="#fff" opacity=".2" />
                        <path d="M4 8h30V5H4v3zm0 11h9v-3H4v3zm0-5h30v-1H4v1zm24 5h6v-3h-6v3z" fill="#fff" />
                    </svg>
                    <svg class="h-8 w-10 text-gray-500" viewBox="0 0 38 24" fill="currentColor">
                        <path d="M35 0H3C1.3 0 0 1.3 0 3v18c0 1.7 1.4 3 3 3h32c1.7 0 3-1.3 3-3V3c0-1.7-1.4-3-3-3z"
                            opacity=".1" />
                        <path
                            d="M35 1c1.1 0 2 .9 2 2v18c0 1.1-.9 2-2 2H3c-1.1 0-2-.9-2-2V3c0-1.1.9-2 2-2h32m0-1H3C1.3 0 0 1.3 0 3v18c0 1.7 1.4 3 3 3h32c1.7 0 3-1.3 3-3V3c0-1.7-1.4-3-3-3z"
                            fill="#fff" opacity=".2" />
                        <path
                            d="M12 12.5c0-2.8 2-4.5 5-4.5 2 0 3.5 1 4 2.5l-1.5 1c-.5-1-1.5-1.5-2.5-1.5-1.5 0-2.5 1-2.5 2.5s1 2.5 2.5 2.5c1 0 2-.5 2.5-1.5l1.5 1c-.5 1.5-2 2.5-4 2.5-3 0-5-1.7-5-4.5zm10.5 0c0-2.8 2-4.5 5-4.5 2 0 3.5 1 4 2.5l-1.5 1c-.5-1-1.5-1.5-2.5-1.5-1.5 0-2.5 1-2.5 2.5s1 2.5 2.5 2.5c1 0 2-.5 2.5-1.5l1.5 1c-.5 1.5-2 2.5-4 2.5-3 0-5-1.7-5-4.5z"
                            fill="#fff" />
                    </svg>
                </div>
            </div>
        </div>
    </footer>

    <!-- Mobile Bottom Navigation Bar -->
    <div
        class="md:hidden fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 z-50 flex justify-around items-center h-16 safe-area-bottom pb-save">
        <a href="{{ route('home') }}"
            class="flex flex-col items-center justify-center w-full h-full text-gray-500 hover:text-indigo-600 {{ Route::currentRouteName() == 'home' ? 'text-indigo-600' : '' }}">
            <svg class="w-6 h-6 mb-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                    d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
            </svg>
            <span class="text-[10px] font-medium uppercase tracking-wide">Home</span>
        </a>
        <a href="{{ route('shop.index') }}"
            class="flex flex-col items-center justify-center w-full h-full text-gray-500 hover:text-indigo-600 {{ Route::currentRouteName() == 'shop.index' ? 'text-indigo-600' : '' }}">
            <svg class="w-6 h-6 mb-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                    d="M4 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V6zM14 6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V6zM4 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2v-2zM14 16a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z" />
            </svg>
            <span class="text-[10px] font-medium uppercase tracking-wide">Shop</span>
        </a>
        <a href="{{ route('cart.index') }}"
            class="flex flex-col items-center justify-center w-full h-full text-gray-500 hover:text-indigo-600 relative {{ Route::currentRouteName() == 'cart.index' ? 'text-indigo-600' : '' }}">
            <div class="relative">
                <svg class="w-6 h-6 mb-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                </svg>
                @php
                    $cartCount = 0;
                    if (Auth::check()) {
                        $cart = \App\Models\Cart::where('user_id', Auth::id())->first();
                        if ($cart)
                            $cartCount = $cart->items->sum('quantity');
                    } else {
                        $sessId = session('cart_session_id');
                        if ($sessId) {
                            $cart = \App\Models\Cart::where('session_id', $sessId)->first();
                            if ($cart)
                                $cartCount = $cart->items->sum('quantity');
                        }
                    }
                @endphp
                @if($cartCount > 0)
                    <span
                        class="absolute -top-1 -right-1 bg-red-600 text-white text-[10px] w-4 h-4 flex items-center justify-center rounded-full border border-white">{{ $cartCount }}</span>
                @endif
            </div>
            <span class="text-[10px] font-medium uppercase tracking-wide">Cart</span>
        </a>
        <a href="{{ Auth::check() ? route('user.dashboard') : route('login') }}"
            class="flex flex-col items-center justify-center w-full h-full text-gray-500 hover:text-indigo-600 {{ Route::currentRouteName() == 'user.dashboard' ? 'text-indigo-600' : '' }}">
            <svg class="w-6 h-6 mb-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                    d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
            </svg>
            <span class="text-[10px] font-medium uppercase tracking-wide">Account</span>
        </a>
    </div>

    <script>
        window.toggleWishlist = function (productId) {
            fetch('{{ route('wishlist.toggle') }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ product_id: productId })
            })
                .then(async response => {
                    if (response.status === 401) {
                        window.location.href = '{{ route('login') }}';
                        return;
                    }
                    const contentType = response.headers.get("content-type");
                    if (!contentType || contentType.indexOf("application/json") === -1) {
                        const text = await response.text();
                        console.error('Non-JSON response:', text);
                        throw new Error('Server returned non-JSON response');
                    }
                    if (!response.ok) {
                        const data = await response.json().catch(() => ({}));
                        throw new Error(data.message || 'Server error');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data && data.status) {
                        alert(data.message);

                        // Update Wishlist Badge
                        const badge = document.getElementById('wishlist-count-badge');
                        if (badge) {
                            if (data.count && data.count > 0) {
                                badge.innerText = data.count;
                                badge.classList.remove('hidden');
                            } else {
                                badge.classList.add('hidden');
                            }
                        }
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Something went wrong. Please try again.');
                });
        }
    </script>
    @include('partials.chatbot-widget')
    @stack('scripts')
</body>

</html>