@extends('layouts.admin')

@section('title', 'SEO & Content Settings')

@section('content')
    <div class="sm:flex sm:items-center sm:justify-between mb-8">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">SEO & Content Settings</h1>
            <p class="mt-2 text-sm text-gray-700">Manage Page Content and Schema Markup for your store.</p>
        </div>
    </div>

    <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg"
         x-data='seoEditor({
             pages: {{ $pages->toJson() }},
             settings: {{ $seoSettings->toJson() }}
         })'
         x-init="initEditor()">
        
        <div class="p-6 bg-white border-b border-gray-200">
            @if(session('success'))
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-4" role="alert">
                    <p>{{ session('success') }}</p>
                </div>
            @endif

            <form action="{{ route('admin.settings.seo.update') }}" method="POST" id="seoForm" @submit.prevent="submitForm">
                @csrf

                <div class="mb-6">
                    <label for="page_selector" class="block text-sm font-medium text-gray-700 mb-2">Select Page to Edit</label>
                    <select id="page_selector" x-model="selectedSlug" @change="changePage()"
                        class="block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm rounded-md">
                        <template x-for="page in pages" :key="page.slug">
                            <option :value="page.slug" x-text="page.title"></option>
                        </template>
                    </select>
                </div>

                <div class="space-y-6">
                    <!-- Page Content Editor -->
                    <div>
                        <h3 class="text-lg leading-6 font-medium text-gray-900 mb-2">Page Content</h3>
                        <p class="text-xs text-gray-500 mb-2">This content will be displayed on the selected page.</p>
                        <div class="mt-1">
                            <!-- Helper hidden input to ensure data sync -->
                            <input type="hidden" :name="'page_content[' + selectedSlug + ']'" id="page_content_input">
                            <div id="editor"></div>
                        </div>
                    </div>

                    <div class="border-t border-gray-200 pt-6"></div>

                    <!-- Schema Markup -->
                    <div>
                        <h3 class="text-lg leading-6 font-medium text-gray-900 mb-2">Schema Markup (JSON-LD)</h3>
                        <div class="mt-1">
                            <label :for="'schema_' + selectedSlug" class="block text-sm font-medium text-gray-700 sr-only">Schema</label>
                            <textarea 
                                :name="'schemas[' + selectedSlug + ']'" 
                                :id="'schema_' + selectedSlug"
                                x-model="currentSchema"
                                rows="10"
                                class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md font-mono text-xs"
                                placeholder='<script type="application/ld+json">...</script>'></textarea>
                            <p class="mt-2 text-xs text-gray-500">Enter valid JSON-LD structure including the script tags.</p>
                        </div>
                    </div>
                </div>

                <div class="mt-8 flex justify-end">
                    <button type="submit"
                        class="bg-indigo-600 border border-transparent rounded-md shadow-sm py-2 px-4 inline-flex justify-center text-sm font-medium text-white hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                        Save Changes
                    </button>
                </div>
            </form>
        </div>
    </div>

    @push('scripts')
    <script src="https://cdn.ckeditor.com/ckeditor5/35.3.2/classic/ckeditor.js"></script>
    <script>
        document.addEventListener('alpine:init', () => {
            Alpine.data('seoEditor', ({ pages, settings }) => ({
                pages: pages,
                settings: settings,
                selectedSlug: pages.length > 0 ? pages[0].slug : '',
                currentSchema: '',
                editorInstance: null,

                initEditor() {
                    if (!this.selectedSlug) return;

                    // Initialize Data
                    this.loadCurrentData();

                    ClassicEditor
                        .create(document.querySelector('#editor'), {
                            toolbar: ['heading', '|', 'bold', 'italic', 'link', 'bulletedList', 'numberedList', 'blockQuote', 'insertTable', '|', 'undo', 'redo'],
                            removePlugins: ['MediaEmbed'] // Remove if not needed or causing issues
                        })
                        .then(editor => {
                            this.editorInstance = editor;
                            
                            // Set initial data
                            const initialContent = this.settings[this.selectedSlug] ? (this.settings[this.selectedSlug].page_content || '') : '';
                            editor.setData(initialContent);

                            // Sync data on change if needed, but we do it on submit mostly
                        })
                        .catch(error => {
                            console.error(error);
                        });
                },

                changePage() {
                    // Save temporary state? No, we reload from 'settings' object unless we want to keep unsaved changes in memory.
                    // For simplicity, we assume unsaved changes are lost if switching pages (standard behavior) OR we could track them.
                    // Let's reload from source of truth (DB data passed in 'settings')
                    
                    this.loadCurrentData();

                    if (this.editorInstance) {
                        const content = this.settings[this.selectedSlug] ? (this.settings[this.selectedSlug].page_content || '') : '';
                        this.editorInstance.setData(content);
                    }
                },

                loadCurrentData() {
                    const setting = this.settings[this.selectedSlug];
                    this.currentSchema = setting ? (setting.schema_markup || '') : '';
                    // Content is handled by CKEditor setData in changePage/init
                },

                submitForm() {
                    // Sync CKEditor data to hidden input
                    const contentValues = this.editorInstance.getData();
                    document.getElementById('page_content_input').value = contentValues;

                    // Let default submission happen or submit programmatically
                     document.getElementById('seoForm').submit();
                }
            }));
        });
    </script>
    @endpush
@endsection