<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order #{{ $order->order_number }} - Lugani Admin</title>
    @vite(['resources/css/app.css', 'resources/js/app.js'])
    <link
        href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Outfit:wght@300;400;500;600;700;800;900&display=swap"
        rel="stylesheet">
    @include('admin.partials.styles')
    <script defer src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <style>
        .font-setting-active {
            border-color: #4f46e5;
            background-color: #f5f3ff;
            color: #4f46e5;
        }

        #page-content {
            transition: font-size 0.2s ease, font-family 0.2s ease;
        }
    </style>
</head>

<body class="bg-gray-50 flex h-screen overflow-hidden">
    @include('admin.partials.sidebar')
    <div class="flex-1 flex flex-col h-screen overflow-hidden">
        <header class="h-16 flex items-center justify-between px-6 bg-white border-b border-gray-200 z-50">
            <h2 class="text-xl font-bold text-gray-800 tracking-tight">Order Details</h2>

            <div class="flex items-center gap-4">
                <!-- Typography Settings Popover -->
                <div class="relative" x-data="{ open: false }">
                    <button @click="open = !open"
                        class="p-2 text-gray-400 hover:text-indigo-600 hover:bg-gray-50 rounded-xl transition-all border border-transparent hover:border-gray-100">
                        <svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4" />
                        </svg>
                    </button>
                    <div x-show="open" @click.away="open = false" x-transition
                        class="absolute right-0 mt-3 w-64 bg-white shadow-2xl rounded-2xl border border-gray-100 p-5 z-[100]">
                        <p class="text-[10px] font-bold text-gray-400 uppercase tracking-widest mb-4">Typography
                            Settings</p>
                        <div class="space-y-4">
                            <div>
                                <label class="text-[9px] font-bold text-gray-500 uppercase mb-2 block">Font
                                    Family</label>
                                <div class="grid grid-cols-2 gap-2 text-center text-[10px] font-bold">
                                    <button onclick="changeFont('Inter')"
                                        class="p-2 border rounded-lg hover:bg-gray-50">Inter</button>
                                    <button onclick="changeFont('Outfit')"
                                        class="p-2 border rounded-lg hover:bg-gray-50">Outfit</button>
                                </div>
                            </div>
                            <div>
                                <label class="text-[9px] font-bold text-gray-500 uppercase mb-2 block">Font
                                    Size</label>
                                <div class="grid grid-cols-3 gap-2 text-center text-[10px] font-bold">
                                    <button onclick="changeSize('0.875rem')"
                                        class="p-2 border rounded-lg hover:bg-gray-50">S</button>
                                    <button onclick="changeSize('1rem')"
                                        class="p-2 border-2 border-indigo-600 rounded-lg bg-indigo-50 text-indigo-600">M</button>
                                    <button onclick="changeSize('1.125rem')"
                                        class="p-2 border rounded-lg hover:bg-gray-50">L</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </header>

        <main class="flex-1 overflow-y-auto bg-gray-50" id="page-content">
            <div class="max-w-[1440px] mx-auto p-4 sm:p-8">
                <!-- Notifications -->
                @if(session('success'))
                    <div class="mb-4 bg-green-50 border-l-4 border-green-500 p-4 rounded-r-lg shadow-sm">
                        <div class="flex items-center">
                            <div class="flex-shrink-0">
                                <svg class="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd"
                                        d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                        clip-rule="evenodd" />
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-green-800">{{ session('success') }}</p>
                            </div>
                        </div>
                    </div>
                @endif
                @if(session('error'))
                    <div class="mb-4 bg-red-50 border-l-4 border-red-500 p-4 rounded-r-lg shadow-sm">
                        <div class="flex items-center">
                            <div class="flex-shrink-0">
                                <svg class="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd"
                                        d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"
                                        clip-rule="evenodd" />
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-red-800">{{ session('error') }}</p>
                            </div>
                        </div>
                    </div>
                @endif

                <!-- Header Stats/Action Bar -->
                <div class="mb-8 flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                    <div class="flex items-center gap-4">
                        <a href="{{ $order->user && $order->user->hasRole('Reseller') ? route('admin.orders.index', ['role' => 'Reseller']) : route('admin.orders.index') }}"
                            class="p-2 bg-white rounded-xl shadow-sm border border-gray-200 text-gray-400 hover:text-indigo-600 transition-colors">
                            <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                            </svg>
                        </a>
                        <div>
                            <h1 class="text-2xl font-bold text-gray-900 tracking-tight">Order
                                #{{ $order->order_number }}</h1>
                            <p class="text-[11px] font-bold text-gray-500 uppercase tracking-[0.15em] mt-1.5">Order
                                Details & Management</p>
                        </div>
                    </div>

                    <div class="flex items-center gap-3 w-full md:w-auto">
                        <a href="{{ route('admin.orders.invoice', $order->id) }}"
                            class="flex items-center gap-2 bg-white border border-gray-200 text-gray-700 font-bold px-4 py-2.5 rounded-xl shadow-sm hover:bg-gray-50 transition-all text-sm">
                            <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                            </svg>
                            Invoice
                        </a>
                        <form action="{{ route('admin.orders.update-status', $order->id) }}" method="POST"
                            class="flex items-center gap-2 flex-1 md:flex-initial">
                            @csrf
                            <select name="status"
                                class="block w-full pl-4 pr-10 py-2.5 text-sm font-bold border-gray-200 focus:ring-indigo-500 focus:border-indigo-500 rounded-xl shadow-sm">
                                <option value="pending" {{ $order->status == 'pending' ? 'selected' : '' }}>Pending
                                </option>
                                <option value="processing" {{ $order->status == 'processing' ? 'selected' : '' }}>
                                    Processing</option>
                                <option value="shipped" {{ $order->status == 'shipped' ? 'selected' : '' }}>Shipped
                                </option>
                                <option value="completed" {{ $order->status == 'completed' ? 'selected' : '' }}>Completed
                                </option>
                                <option value="cancelled" {{ $order->status == 'cancelled' ? 'selected' : '' }}>Cancelled
                                </option>
                            </select>
                            <button type="submit"
                                class="bg-indigo-600 hover:bg-indigo-700 text-white p-2.5 rounded-xl shadow-indigo-200 shadow-lg transition-all active:scale-95">
                                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24"
                                    stroke="currentColor">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M5 13l4 4L19 7" />
                                </svg>
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Timeline Section -->
                @php
                    $steps = [
                        ['status' => 'pending', 'label' => 'Placed', 'icon' => 'M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z'],
                        ['status' => 'processing', 'label' => 'Processed', 'icon' => 'M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4'],
                        ['status' => 'shipped', 'label' => 'In Transit', 'icon' => 'M9 17a2 2 0 11-4 0 2 2 0 014 0zM19 17a2 2 0 11-4 0 2 2 0 014 0z M13 16V6a1 1 0 00-1-1H4a1 1 0 00-1 1v10a1 1 0 001 1h1m8-1a1 1 0 011 1v2a1 1 0 01-1 1m0-4h3m-11 0h1m11 0a1 1 0 001-1V7a1 1 0 00-1-1h-3.3a1 1 0 00-1 1v9h3.3z'],
                        ['status' => 'completed', 'label' => 'Delivered', 'icon' => 'M5 13l4 4L19 7'],
                    ];
                    $histories = $order->statusHistories->keyBy('status');
                    $isCancelled = $order->status === 'cancelled';

                    // Handle Returns
                    if ($order->returnRequest) {
                        $ret = $order->returnRequest;

                        // Add Return Requested Step
                        $steps[] = ['status' => 'return_requested', 'label' => 'Return Req.', 'icon' => 'M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6'];
                        $histories->put('return_requested', $ret);

                        if (in_array($ret->status, ['approved', 'received', 'refund_initiated', 'refunded'])) {
                            $steps[] = ['status' => 'return_approved', 'label' => 'Return Appr.', 'icon' => 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'];
                            $histories->put('return_approved', (object) ['created_at' => $ret->updated_at]);
                        }

                        if ($ret->status == 'rejected') {
                            $steps[] = ['status' => 'return_rejected', 'label' => 'Return Rejected', 'icon' => 'M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z'];
                            $histories->put('return_rejected', (object) ['created_at' => $ret->updated_at]);
                        }

                        if ($ret->status == 'refunded') {
                            $steps[] = ['status' => 'refunded', 'label' => 'Refunded', 'icon' => 'M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z'];
                            $histories->put('refunded', (object) ['created_at' => $ret->refunded_at]);
                        }
                    }

                    $furthestStep = -1;
                    foreach ($steps as $index => $step) {
                        if ($histories->has($step['status']))
                            $furthestStep = $index;
                    }

                    $totalShownSteps = count($steps) + ($isCancelled ? 1 : 0);
                    $progressPercentage = ($totalShownSteps > 1) ? ($furthestStep / ($totalShownSteps - 1)) : 0;
                    $progressWidth = $progressPercentage * 100;

                    $statusLabelColors = [
                        'pending' => 'text-indigo-600',
                        'processing' => 'text-blue-600',
                        'shipped' => 'text-amber-600',
                        'completed' => 'text-green-600',
                        'return_requested' => 'text-yellow-600',
                        'return_approved' => 'text-blue-600',
                        'return_rejected' => 'text-red-600',
                        'refunded' => 'text-purple-600',
                    ];
                @endphp

                <div class="bg-white shadow-xl rounded-[2rem] mb-10 border border-gray-100 overflow-visible">
                    <div class="p-8 sm:p-12 max-w-5xl mx-auto w-full">
                        <div class="relative px-6">
                            <!-- Precise Center-to-Center Bar -->
                            <div class="absolute top-6 left-6 right-6 h-1 bg-gray-100 rounded-full overflow-hidden">
                                @if(!$isCancelled)
                                    <div class="h-full bg-gradient-to-r from-indigo-500 to-blue-500 transition-all duration-1000"
                                        style="width: {{ $progressWidth }}%"></div>
                                @endif
                            </div>

                            <div class="flex justify-between relative z-10">
                                @foreach ($steps as $index => $step)
                                    @php
                                        $active = $histories->has($step['status']);
                                        $isCurrent = $order->status === $step['status'];
                                        // Note: For returns, strict equality check on $order->status might not capture "current" properly since status might be 'completed' while return is 'requested'. 
                                        // But this matches the loop logic. The visual 'active' state is driven by $active (history existence).

                                        $labelColor = $active ? ($statusLabelColors[$step['status']] ?? 'text-gray-900') : 'text-gray-400';
                                        $iconClasses = $active
                                            ? 'bg-indigo-600 border-indigo-600 text-white shadow-lg scale-110'
                                            : 'bg-white border-gray-200 text-gray-300';

                                        // Custom styling for return steps to make them distinct
                                        if ($active && str_contains($step['status'], 'return')) {
                                            $iconClasses = 'bg-blue-600 border-blue-600 text-white shadow-lg scale-110';
                                        }
                                        if ($active && $step['status'] == 'return_rejected') {
                                            $iconClasses = 'bg-red-500 border-red-500 text-white shadow-lg scale-110';
                                        }
                                        if ($active && $step['status'] == 'refunded') {
                                            $iconClasses = 'bg-purple-600 border-purple-600 text-white shadow-lg scale-110';
                                        }
                                    @endphp
                                    <div class="flex flex-col items-center flex-1">
                                        <div
                                            class="w-12 h-12 rounded-2xl flex items-center justify-center border-2 transition-all duration-500 {{ $iconClasses }}">
                                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="{{ $step['icon'] }}"></path>
                                            </svg>
                                        </div>
                                        <div class="mt-4 text-center">
                                            <span
                                                class="block text-[11px] font-bold uppercase tracking-[0.15em] {{ $labelColor }}">
                                                {{ $step['label'] }}
                                            </span>
                                            @if($active && ($hist = $histories->get($step['status'])))
                                                <p class="text-[10px] text-gray-500 font-bold mt-1 uppercase">
                                                    {{ $hist->created_at->format('M d • h:i A') }}
                                                </p>
                                            @endif
                                        </div>
                                    </div>
                                @endforeach

                                @if($isCancelled)
                                    <div class="flex flex-col items-center flex-1">
                                        <div
                                            class="w-12 h-12 rounded-2xl flex items-center justify-center border-2 bg-red-600 border-red-600 text-white shadow-lg animate-pulse">
                                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                        </div>
                                        <div class="mt-4 text-center text-red-600">
                                            <span
                                                class="block text-[10px] font-bold uppercase tracking-[0.1em]">Cancelled</span>
                                            @if($hist = $histories->get('cancelled'))
                                                <p class="text-[10px] text-red-500 font-bold mt-1 uppercase">
                                                    {{ $hist->created_at->format('M d • h:i A') }}
                                                </p>
                                            @endif
                                        </div>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>

                @if($order->status === 'cancelled' && $order->cancellation_reason)
                    <div class="mb-8 bg-red-50 border-2 border-red-100 rounded-[2rem] p-8 flex items-start gap-6 shadow-sm">
                        <div
                            class="h-12 w-12 rounded-2xl bg-red-600 flex items-center justify-center text-white flex-shrink-0 shadow-lg shadow-red-200">
                            <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24"
                                stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                            </svg>
                        </div>
                        <div>
                            <h4 class="text-sm font-bold text-red-900 uppercase tracking-widest mb-1">Cancellation Reason
                            </h4>
                            <p class="text-red-700 font-medium leading-relaxed">{{ $order->cancellation_reason }}</p>
                            @if($hist = $order->statusHistories->where('status', 'cancelled')->first())
                                <p class="text-[10px] text-red-400 font-bold uppercase mt-3 tracking-wider">
                                    Cancelled on {{ $hist->created_at->format('F d, Y \a\t h:i A') }}
                                </p>
                            @endif
                        </div>
                    </div>
                @endif

                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Left Column: Content -->
                    <div class="lg:col-span-2 space-y-8">
                        <div class="bg-white shadow-xl rounded-[2rem] border border-gray-100 overflow-hidden" x-data="{
                                items: [
                                    @foreach($order->items as $item)
                                        {
                                            id: {{ $item->id }},
                                            qty: {{ $item->quantity }},
                                            price: {{ $item->price }},
                                            discount_percent: {{ $item->options['discount_percentage'] ?? 0 }},
                                            get total() {
                                                let discountedPrice = this.price * (1 - (this.discount_percent / 100));
                                                return discountedPrice * this.qty;
                                            }
                                        },
                                    @endforeach
                                ],
                                overall_discount_percent: 0,
                                get subtotal() {
                                    let rawSubtotal = this.items.reduce((sum, item) => sum + item.total, 0);
                                    return rawSubtotal * (1 - (this.overall_discount_percent / 100));
                                },
                                get total() {
                                    // Subtotal (already discounted if reseller) + Shipping
                                    let t = this.subtotal + {{ $order->shipping_cost }};
                                    // Normal Discount
                                    t -= {{ $order->discount ?? 0 }};
                                    // Wallet
                                    t -= {{ $order->points_discount ?? 0 }};
                                    // Gift Card
                                    t -= {{ $order->gift_card_discount ?? 0 }};
                                    return Math.max(0, t);
                                }
                             }">
                            <div
                                class="px-8 py-6 border-b border-gray-100 flex justify-between items-center bg-gray-50/80">
                                <h3 class="text-sm font-bold uppercase tracking-[0.2em] text-gray-600">Order Items</h3>
                                <div class="flex gap-2 items-center">
                                    <span
                                        class="bg-indigo-600 text-white text-[10px] font-bold px-4 py-1.5 rounded-full uppercase shadow-sm">
                                        {{ $order->items->count() }} Items
                                    </span>
                                    @if(request('role') === 'Reseller')
                                        @if($order->status == 'quotation_sent')
                                            <span
                                                class="bg-blue-100 text-blue-800 text-[10px] font-bold px-3 py-1 rounded-full uppercase border border-blue-200">
                                                Quotation Sent
                                            </span>
                                        @elseif($order->status == 'quotation_accepted')
                                            <span
                                                class="bg-green-100 text-green-800 text-[10px] font-bold px-3 py-1 rounded-full uppercase border border-green-200">
                                                Quotation Accepted
                                            </span>
                                        @endif
                                    @endif
                                </div>
                            </div>

                            @php
                                $isResellerPending = request('role') === 'Reseller' && $order->status === 'pending';
                            @endphp

                            @if($isResellerPending)
                                <form action="{{ route('admin.orders.generate-quotation', $order->id) }}" method="POST"
                                    id="quotation-form">
                                    @csrf
                            @endif

                                <div class="overflow-x-auto">
                                    <table class="min-w-full divide-y divide-gray-100">
                                        <thead class="bg-gray-50/50">
                                            <tr>
                                                <th
                                                    class="px-8 py-5 text-left text-[11px] font-bold text-gray-600 uppercase tracking-[0.1em]">
                                                    Product</th>
                                                <th
                                                    class="px-8 py-5 text-left text-[11px] font-bold text-gray-600 uppercase tracking-[0.1em]">
                                                    Price</th>
                                                @if($isResellerPending)
                                                    <th
                                                        class="px-8 py-5 text-left text-[11px] font-bold text-gray-600 uppercase tracking-[0.1em]">
                                                        Disc. %</th>
                                                @endif
                                                <th
                                                    class="px-8 py-5 text-left text-[11px] font-bold text-gray-600 uppercase tracking-[0.1em]">
                                                    Qty</th>
                                                <th
                                                    class="px-8 py-5 text-right text-[11px] font-bold text-gray-600 uppercase tracking-[0.1em]">
                                                    Subtotal</th>
                                            </tr>
                                        </thead>
                                        <tbody class="divide-y divide-gray-50">
                                            @foreach($order->items as $index => $item)
                                                <tr class="hover:bg-gray-50/50 transition-colors">
                                                    <td class="px-8 py-5 flex items-center gap-4">
                                                        @php
                                                            $imgPath = $item->variation->image_path ?? $item->product->images->first()->image_path ?? null;
                                                        @endphp
                                                        <div
                                                            class="h-12 w-12 rounded-xl overflow-hidden bg-gray-100 border border-gray-100 flex-shrink-0">
                                                            @if($imgPath)
                                                                <img src="{{ $imgPath }}" class="h-full w-full object-cover">
                                                            @else
                                                                <div
                                                                    class="h-full w-full flex items-center justify-center text-[10px] font-bold text-gray-300">
                                                                    N/A</div>
                                                            @endif
                                                        </div>
                                                        <div>
                                                            <p class="text-sm font-bold text-gray-900 leading-tight">
                                                                {{ $item->product->name ?? 'Deleted Product' }}
                                                            </p>
                                                            @if(isset($item->options['attributes']))
                                                                <div class="flex gap-2 mt-1">
                                                                    @foreach($item->options['attributes'] as $attr)
                                                                        <span
                                                                            class="text-[9px] font-bold bg-gray-100 text-gray-500 px-2 py-0.5 rounded-md uppercase">{{ $attr['attribute'] }}:
                                                                            {{ $attr['value'] }}</span>
                                                                    @endforeach
                                                                </div>
                                                            @endif
                                                        </div>
                                                    </td>
                                                    <td class="px-8 py-5 text-sm font-bold text-gray-500">
                                                        @if($isResellerPending)
                                                            <div class="flex items-center">
                                                                <span class="mr-1 text-gray-400">₹</span>
                                                                <input type="number" step="0.01"
                                                                    name="items[{{ $item->id }}][price]"
                                                                    x-model="items[{{ $index }}].price"
                                                                    class="w-24 p-1 text-sm border-gray-200 rounded-md focus:ring-indigo-500 focus:border-indigo-500">
                                                            </div>
                                                        @else
                                                            ₹{{ number_format($item->price, 2) }}
                                                        @endif
                                                    </td>
                                                    @if($isResellerPending)
                                                        <td class="px-8 py-5 text-sm font-bold text-gray-500">
                                                            <input type="number" step="0.1" min="0" max="100"
                                                                name="items[{{ $item->id }}][discount_percent]"
                                                                x-model="items[{{ $index }}].discount_percent"
                                                                class="w-16 p-1 text-sm border-gray-200 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
                                                                placeholder="0">
                                                        </td>
                                                    @endif
                                                    <td class="px-8 py-5 text-sm font-bold text-gray-900">
                                                        {{ $item->quantity }}</td>
                                                    <td class="px-8 py-5 text-right text-sm font-bold text-gray-900">
                                                        <span x-text="'₹' + items[{{ $index }}].total.toFixed(2)"></span>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>

                                <div class="p-8 bg-gray-50/30 border-t border-gray-100">
                                    <div class="flex flex-col md:flex-row justify-between items-end gap-6">
                                        @if($isResellerPending)
                                            <div class="w-full md:w-auto">
                                                <button type="submit"
                                                    class="bg-indigo-600 text-white px-6 py-3 rounded-xl font-bold uppercase tracking-wider text-xs hover:bg-indigo-700 shadow-lg transition-all w-full md:w-auto">
                                                    Generate & Send Quotation
                                                </button>
                                                <p
                                                    class="text-[10px] text-gray-400 mt-2 font-medium max-w-xs text-center md:text-left">
                                                    This will update prices and initiate the quotation flow.
                                                </p>
                                            </div>
                                        @else
                                            <div></div>
                                        @endif

                                        <div class="max-w-xs w-full space-y-3">
                                            {{-- Subtotal --}}
                                            <div
                                                class="flex justify-between text-xs font-bold text-gray-500 uppercase tracking-widest">
                                                <span>Subtotal</span>
                                                <span class="text-gray-900 font-bold"
                                                    x-text="'₹' + subtotal.toFixed(2)">₹{{ number_format($order->subtotal, 2) }}</span>
                                            </div>

                                            {{-- Shipping --}}
                                            <div
                                                class="flex justify-between text-xs font-bold text-gray-500 uppercase tracking-widest">
                                                <span>Shipping</span>
                                                <span
                                                    class="text-gray-900 font-bold">₹{{ number_format($order->shipping_cost, 2) }}</span>
                                            </div>

                                            {{-- Reseller Overall Discount Input --}}
                                            @if($isResellerPending)
                                                <div
                                                    class="flex justify-between text-xs font-bold text-gray-500 uppercase tracking-widest items-center">
                                                    <span>Overall Discount (%)</span>
                                                    <input type="number" step="0.1" min="0" max="100"
                                                        name="overall_discount_percent" x-model="overall_discount_percent"
                                                        class="w-16 p-1 text-sm text-right border-gray-200 rounded-md focus:ring-indigo-500 focus:border-indigo-500"
                                                        placeholder="0">
                                                </div>
                                            @endif

                                            {{-- Coupon Discount --}}
                                            @if($order->discount > 0 && !$isResellerPending)
                                                <div
                                                    class="flex justify-between text-xs font-bold text-emerald-600 uppercase tracking-wider">
                                                    <div class="flex flex-col">
                                                        <span>Discount</span>
                                                        @if($order->coupon_code)
                                                            <span
                                                                class="text-[9px] text-gray-400">{{ $order->coupon_code }}</span>
                                                        @endif
                                                    </div>
                                                    <span>-₹{{ number_format($order->discount, 2) }}</span>
                                                </div>
                                            @endif

                                            {{-- Points / Wallet --}}
                                            @if($order->points_discount > 0)
                                                <div
                                                    class="flex justify-between text-xs font-bold text-indigo-500 uppercase tracking-wider">
                                                    <div class="flex flex-col">
                                                        <span>Wallet Used</span>
                                                        <span class="text-[9px] text-gray-400">{{ $order->points_redeemed }}
                                                            Pts</span>
                                                    </div>
                                                    <span>-₹{{ number_format($order->points_discount, 2) }}</span>
                                                </div>
                                            @endif

                                            {{-- Gift Card --}}
                                            @if($order->gift_card_discount > 0)
                                                <div
                                                    class="flex justify-between text-xs font-bold text-purple-500 uppercase tracking-wider">
                                                    <span>Gift Card</span>
                                                    <span>-₹{{ number_format($order->gift_card_discount, 2) }}</span>
                                                </div>
                                            @endif

                                            {{-- Final Total --}}
                                            <div
                                                class="flex justify-between text-lg font-bold text-gray-900 pt-4 border-t border-gray-200">
                                                <span>TOTAL</span>
                                                <span
                                                    x-text="'₹' + total.toFixed(2)">₹{{ number_format($order->total, 2) }}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                @if($isResellerPending)
                                    </form>
                                @endif

                        </div>

                        {{-- Final Ratings Section --}}
                        @if($order->agentRating || $order->productReviews->isNotEmpty())
                            <div id="ratings-section"
                                class="bg-white shadow-xl rounded-[2rem] border border-gray-100 overflow-hidden">
                                <div class="px-8 py-6 border-b border-gray-100 bg-amber-50/50">
                                    <h3
                                        class="text-sm font-bold uppercase tracking-[0.2em] text-amber-700 flex items-center">
                                        <svg class="w-5 h-5 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                            <path
                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                        </svg>
                                        Customer Feedback & Reviews
                                    </h3>
                                </div>
                                <div class="p-8 space-y-8">
                                    @if($order->agentRating)
                                        <div class="bg-amber-50 p-6 rounded-2xl border border-amber-100">
                                            <div class="flex justify-between items-center mb-4">
                                                <div>
                                                    <p
                                                        class="text-[10px] font-bold text-amber-600 uppercase tracking-widest leading-none mb-1">
                                                        Delivery Performance</p>
                                                    <h4 class="text-lg font-bold text-gray-900">Driver:
                                                        {{ $order->deliveryAgent->user->name ?? 'N/A' }}</h4>
                                                </div>
                                                <div class="flex">
                                                    @for($i = 1; $i <= 5; $i++)
                                                        <svg class="w-5 h-5 {{ $i <= $order->agentRating->rating ? 'text-amber-500 fill-current' : 'text-gray-200 fill-current' }}"
                                                            viewBox="0 0 20 20">
                                                            <path
                                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                        </svg>
                                                    @endfor
                                                </div>
                                            </div>
                                            @if($order->agentRating->comment)
                                                <p
                                                    class="text-gray-700 italic font-medium bg-white/50 p-4 rounded-xl border border-amber-50 leading-relaxed text-sm">
                                                    "{{ $order->agentRating->comment }}"</p>
                                            @endif
                                        </div>
                                    @endif

                                    @if($order->productReviews->isNotEmpty())
                                        <div class="space-y-6">
                                            <p class="text-xs font-bold text-gray-400 uppercase tracking-[0.2em] mb-4">Product
                                                Experience</p>
                                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                @foreach($order->productReviews as $review)
                                                    <div class="bg-gray-50 border border-gray-100 rounded-2xl p-5">
                                                        <div class="flex justify-between items-start mb-3">
                                                            <div class="flex items-center gap-3">
                                                                <div
                                                                    class="h-10 w-10 bg-white rounded-lg border border-gray-100 p-1 flex-shrink-0">
                                                                    <img src="{{ $review->product->images->first()->image_path ?? 'https://placehold.co/100x100?text=No+Image' }}"
                                                                        class="h-full w-full object-cover rounded-md">
                                                                </div>
                                                                <p class="text-sm font-bold text-gray-900 line-clamp-1">
                                                                    {{ $review->product->name }}</p>
                                                            </div>
                                                            <div class="flex">
                                                                @for($i = 1; $i <= 5; $i++)
                                                                    <svg class="w-3 h-3 {{ $i <= $review->rating ? 'text-amber-500 fill-current' : 'text-gray-200 fill-current' }}"
                                                                        viewBox="0 0 20 20">
                                                                        <path
                                                                            d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                                    </svg>
                                                                @endfor
                                                            </div>
                                                        </div>
                                                        @if($review->comment)
                                                            <p
                                                                class="text-[13px] text-gray-600 italic bg-white/30 p-3 rounded-lg border border-gray-50">
                                                                "{{ $review->comment }}"</p>
                                                        @endif
                                                    </div>
                                                @endforeach
                                            </div>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        @endif
                    </div>

                    <!-- Right Column: Sidebar -->
                    <div class="space-y-8">
                        @if(request('role') !== 'Reseller')
                            <!-- Customer -->
                            <div
                                class="bg-slate-900 shadow-xl rounded-[2.5rem] p-8 text-white relative overflow-hidden ring-1 ring-white/10">
                                <div class="absolute -top-12 -right-12 w-32 h-32 bg-white/5 rounded-full blur-3xl">
                                </div>
                                <p class="text-[11px] font-bold uppercase tracking-[0.2em] text-slate-300 mb-6">Customer
                                    Profile</p>
                                <div class="flex items-center gap-4 mb-8">
                                    <div
                                        class="h-14 w-14 rounded-2xl bg-slate-700/50 flex items-center justify-center font-bold text-2xl border border-white/10 shadow-inner">
                                        {{ substr($order->name, 0, 1) }}
                                    </div>
                                    <div>
                                        <p class="text-xl font-bold tracking-tight leading-none">{{ $order->name }}</p>
                                        <p class="text-xs text-slate-300 mt-1.5 font-bold">{{ $order->email }}</p>
                                    </div>
                                </div>
                                <div class="bg-black/20 rounded-2xl p-4 border border-white/5 space-y-3">
                                    <div class="flex justify-between text-xs font-bold">
                                        <span class="text-slate-400 font-bold uppercase tracking-widest">Contact</span>
                                        <span class="text-white">{{ $order->phone }}</span>
                                    </div>
                                    <div class="flex justify-between text-xs font-bold">
                                        <span class="text-slate-400 font-bold uppercase tracking-widest">Account</span>
                                        <span class="text-white">{{ $order->user_id ? 'Registered' : 'Guest' }}</span>
                                    </div>
                                </div>
                            </div>
                        @endif

                        <!-- Grid Address -->
                        <div
                            class="bg-slate-900 shadow-xl rounded-[2.5rem] p-8 text-white relative overflow-hidden ring-1 ring-white/10">
                            <p class="text-[11px] font-bold uppercase tracking-[0.2em] text-slate-300 mb-6">
                                Delivery Location</p>
                            @if($order->shippingAddress)
                                <div class="space-y-2 text-sm font-bold text-slate-200 leading-relaxed">
                                    <p class="text-white text-lg font-bold mb-4">{{ $order->shippingAddress->name }}</p>
                                    <p>{{ $order->shippingAddress->address_line1 }}</p>
                                    @if($order->shippingAddress->address_line2)
                                    <p>{{ $order->shippingAddress->address_line2 }}</p> @endif
                                    <p>{{ $order->shippingAddress->city }}, {{ $order->shippingAddress->state }}</p>
                                    class="text-indigo-300 font-bold mt-2 inline-block bg-white/5 px-3 py-1 rounded-full
                                    text-xs box-shadow ring-1 ring-white/10">
                                    {{ $order->shippingAddress->country }} / {{ $order->shippingAddress->postal_code }}
                                    </p>
                                </div>
                                <div class="mt-6 pt-4 border-t border-white/5 flex items-center gap-2">
                                    <span
                                        class="text-[10px] font-bold text-slate-400 uppercase tracking-widest">Phone:</span>
                                    <span
                                        class="text-xs font-bold text-slate-200">{{ $order->shippingAddress->phone }}</span>
                                </div>
                            @else
                                <p class="text-sm font-bold text-slate-500 italic">No physical address found.</p>
                            @endif
                        </div>

                        <!-- Payment -->
                        <div
                            class="bg-slate-900 shadow-xl rounded-[2.5rem] p-8 text-white relative overflow-hidden ring-1 ring-white/10">
                            <p class="text-[11px] font-bold uppercase tracking-[0.2em] text-slate-300 mb-6">
                                Payment Summary</p>
                            <div class="flex justify-between items-end">
                                <div>
                                    <p
                                        class="text-xs font-bold text-slate-400 uppercase tracking-widest opacity-80 mb-1">
                                        Method</p>
                                    <p class="text-xl font-bold text-white tracking-tight">
                                        {{ $order->payment_method ?? 'Unknown' }}
                                    </p>
                                </div>
                                <span
                                    class="bg-white/10 border border-white/20 text-white px-4 py-1.5 rounded-full text-[10px] font-bold uppercase tracking-widest shadow-lg">
                                    {{ $order->payment_status }}
                                </span>
                            </div>
                            @if($order->razorpay_order_id || $order->transaction_id)
                                <div class="mt-6 bg-black/30 rounded-2xl p-4 border border-white/5">
                                    <p
                                        class="text-[9px] font-bold text-slate-400 hidden sm:block uppercase tracking-widest mb-1 opacity-50">
                                        Transaction Hash</p>
                                    <p class="text-[10px] font-mono font-bold break-all text-slate-100 leading-tight">
                                        {{ $order->razorpay_order_id ?? $order->transaction_id }}
                                    </p>
                                </div>
                            @endif
                        </div>

                        <!-- Delivery Management (Local Only) -->
                        @if($order->order_scope === 'Local' && request('role') !== 'Reseller')
                            <div
                                class="bg-slate-900 shadow-xl rounded-[2.5rem] p-8 text-white relative overflow-hidden ring-1 ring-white/10">
                                <p class="text-[11px] font-bold uppercase tracking-[0.2em] text-slate-300 mb-6">Fleet
                                    Management</p>
                                @if($order->deliveryAgent)
                                    <div class="flex items-center gap-4 mb-6 p-4 bg-white/5 rounded-2xl border border-white/5">
                                        <div
                                            class="h-12 w-12 rounded-xl bg-slate-700/50 flex items-center justify-center font-bold text-xl border border-white/10 shadow-inner">
                                            {{ substr($order->deliveryAgent->user->name ?? 'A', 0, 1) }}
                                        </div>
                                        <div>
                                            <p class="text-base font-bold text-white leading-none">
                                                {{ $order->deliveryAgent->user->name ?? 'Unknown' }}
                                            </p>
                                            <p class="text-[11px] text-slate-400 mt-2 font-bold">
                                                {{ $order->deliveryAgent->phone ?? 'No Phone' }}
                                            </p>
                                        </div>
                                    </div>
                                @endif

                                <form action="{{ route('admin.orders.assign-agent', $order->id) }}" method="POST"
                                    class="space-y-4">
                                    @csrf
                                    @if($order->status == 'cancelled' || $order->status == 'completed' || $order->status == 'returned')
                                        <div
                                            class="bg-red-500/10 border border-red-500/20 text-red-400 text-[10px] font-bold px-4 py-2 rounded-xl uppercase tracking-widest text-center mb-2">
                                            Agent Locked (Order {{ ucfirst($order->status) }})
                                        </div>
                                    @endif

                                    @if($order->order_scope === 'Local' && $order->status === 'pending')
                                        <div>
                                            <label
                                                class="text-[10px] font-bold text-slate-400 uppercase tracking-widest mb-1.5 block">Pickup
                                                Store</label>
                                            <select name="pickup_store_id"
                                                class="w-full rounded-xl border-0 shadow-inner focus:ring-2 focus:ring-white/50 text-xs font-bold p-3 bg-white/10 text-white backdrop-blur-md appearance-none mb-2">
                                                <option value="" class="bg-slate-900">Select Pickup Location...</option>
                                                @foreach($stores as $store)
                                                    <option value="{{ $store->id }}" {{ $order->pickup_store_id == $store->id ? 'selected' : '' }} class="text-slate-900 bg-white font-bold">
                                                        {{ $store->name }}
                                                    </option>
                                                @endforeach
                                            </select>
                                        </div>
                                    @elseif($order->pickupStore)
                                        <div class="p-3 bg-white/5 rounded-xl border border-white/5 mb-2">
                                            <p class="text-[10px] font-bold text-slate-400 uppercase tracking-widest mb-1">
                                                Assigned Pickup Store</p>
                                            <p class="text-xs font-bold text-white">{{ $order->pickupStore->name }}</p>
                                            <p class="text-[10px] text-slate-400 font-medium">{{ $order->pickupStore->city }},
                                                {{ $order->pickupStore->state }}</p>
                                        </div>
                                    @endif

                                    <div class="flex gap-2">
                                        <select name="delivery_agent_id" required @if($order->status == 'completed' || $order->status == 'cancelled' || $order->status == 'returned') disabled @endif
                                            class="flex-1 rounded-xl border-0 shadow-inner focus:ring-2 focus:ring-white/50 text-xs font-bold p-3 bg-white/10 text-white backdrop-blur-md appearance-none">
                                            <option value="" class="bg-slate-900">Assign Agent...</option>
                                            @foreach($deliveryAgents as $agent)
                                                <option value="{{ $agent->id }}" {{ $order->delivery_agent_id == $agent->id ? 'selected' : '' }} class="text-slate-900 bg-white font-bold">
                                                    {{ $agent->user->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @if($order->status !== 'completed' && $order->status !== 'cancelled' && $order->status !== 'returned')
                                            <button type="submit"
                                                class="bg-white text-slate-900 p-3 rounded-xl shadow-lg hover:scale-105 active:scale-95 transition-all">
                                                <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none"
                                                    viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M5 13l4 4L19 7" />
                                                </svg>
                                            </button>
                                        @endif
                                    </div>
                                </form>
                            </div>
                        @endif
                    </div> <!-- End Sidebar -->
                </div> <!-- End Grid -->
            </div> <!-- End Container -->
        </main>
    </div> <!-- End Page Wrapper -->

    <script>
        function changeFont(font) {
            document.getElementById('page-content').style.fontFamily = `"${font}", sans-serif`;
            localStorage.setItem('admin_font_family', font);
            updateActiveStates();
        }

        function changeSize(size) {
            document.getElementById('page-content').style.fontSize = size;
            localStorage.setItem('admin_font_size', size);
            updateActiveStates();
        }

        function updateActiveStates() {
            const currentFont = localStorage.getItem('admin_font_family') || 'Inter';
            const currentSize = localStorage.getItem('admin_font_size') || '1rem';

            // Update font buttons
            document.querySelectorAll('[onclick^="changeFont"]').forEach(btn => {
                if (btn.getAttribute('onclick').includes(currentFont)) {
                    btn.classList.add('font-setting-active', 'ring-2', 'ring-indigo-600', 'bg-indigo-50', 'text-indigo-600');
                } else {
                    btn.classList.remove('font-setting-active', 'ring-2', 'ring-indigo-600', 'bg-indigo-50', 'text-indigo-600');
                }
            });

            // Update size buttons
            document.querySelectorAll('[onclick^="changeSize"]').forEach(btn => {
                if (btn.getAttribute('onclick').includes(currentSize)) {
                    btn.classList.add('font-setting-active', 'ring-2', 'ring-indigo-600', 'bg-indigo-50', 'text-indigo-600');
                } else {
                    btn.classList.remove('font-setting-active', 'ring-2', 'ring-indigo-600', 'bg-indigo-50', 'text-indigo-600');
                }
            });
        }

        // Initialize on load
        document.addEventListener('DOMContentLoaded', function () {
            const savedFont = localStorage.getItem('admin_font_family');
            const savedSize = localStorage.getItem('admin_font_size');
            if (savedFont) changeFont(savedFont);
            if (savedSize) changeSize(savedSize);
            updateActiveStates();
        });
    </script>
</body>

</html>