<?php

namespace App\Services;

use App\Models\EmailTemplate;
use App\Models\GeneralSetting;
use App\Mail\DynamicTemplateMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class UserMailService
{
    public static function sendWelcomeEmail($user)
    {
        self::sendNotification($user, 'Welcome', 'Clients');
    }

    public static function sendCartReminder($user)
    {
        // Check if user has cart items
        $cart = \App\Models\Cart::where('user_id', $user->id)->with('items.product')->first();
        if (!$cart || $cart->items->isEmpty()) {
            return;
        }

        $itemsSummary = self::getCartItemsSummary($cart);
        $data = [
            'customer_name' => $user->name,
            'cart_items_summary' => $itemsSummary,
            'cart_url' => route('cart.index'),
        ];

        self::sendNotification($user, 'Cart Reminder', 'Clients', $data);
    }

    public static function sendWishlistReminder($user)
    {
        // Check if user has wishlist items
        // Assuming a Wishlist model exists or similar logic
        $wishlistItems = \App\Models\Wishlist::where('user_id', $user->id)->with('product')->get();

        if ($wishlistItems->isEmpty()) {
            return;
        }

        $itemsSummary = self::getWishlistItemsSummary($wishlistItems);
        $data = [
            'customer_name' => $user->name,
            'wishlist_items_summary' => $itemsSummary,
            'wishlist_url' => route('user.wishlist'), // Assuming route
        ];

        self::sendNotification($user, 'Wishlist Reminder', 'Clients', $data);
    }

    public static function sendNotification($user, $category, $role, $extraData = [])
    {
        try {
            $template = EmailTemplate::where('category', $category)
                ->where('target_role', $role)
                ->where('is_active', true)
                ->first();

            if (!$template) {
                Log::warning("Email template not found for Category: {$category}, Role: {$role}");
                return;
            }

            // Default Data
            $data = [
                'customer_name' => $user->name,
                'email' => $user->email,
                'app_name' => config('app.name'),
            ];

            // Merge extra data
            $data = array_merge($data, $extraData);

            if ($user->email) {
                Mail::to($user->email)->send(new DynamicTemplateMail($template, $data));
            }
        } catch (\Exception $e) {
            Log::error("Failed to send user email ({$category}): " . $e->getMessage());
        }
    }

    public static function getCartItemsSummary($cart)
    {
        $html = '<ul style="padding-left: 20px;">';
        foreach ($cart->items as $item) {
            $name = $item->product ? $item->product->name : 'Product';
            $qty = $item->quantity;
            $html .= "<li>{$qty} x {$name}</li>";
        }
        $html .= '</ul>';
        return $html;
    }

    public static function getWishlistItemsSummary($items)
    {
        $html = '<ul style="padding-left: 20px;">';
        foreach ($items as $item) {
            $name = $item->product ? $item->product->name : 'Product';
            $html .= "<li>{$name}</li>";
        }
        $html .= '</ul>';
        return $html;
    }
}
