<?php

namespace App\Services;

use App\Models\GeneralSetting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Google\Client as GoogleClient;

class FCMService
{
    protected $serverKey;
    protected $projectId;
    protected $serviceAccount;

    public function __construct()
    {
        $settings = GeneralSetting::first();
        if ($settings) {
            $this->serverKey = $settings->fcm_server_key;
            $this->projectId = $settings->fcm_project_id;
            $this->serviceAccount = $settings->fcm_service_account_json;
        }
    }

    public function sendNotification($token, $title, $body, $data = [])
    {
        if (!$token) {
            return false;
        }

        // 1. Try HTTP v1 (Preferred)
        if ($this->serviceAccount && $this->projectId) {
            return $this->sendHttpV1($token, $title, $body, $data);
        }

        // 2. Fallback to Legacy
        if ($this->serverKey) {
            return $this->sendLegacy($token, $title, $body, $data);
        }

        Log::warning('FCM: No credentials found (Service Account or Server Key). Cannot send notification.');
        return false;
    }

    protected function sendHttpV1($token, $title, $body, $data)
    {
        try {
            $client = new GoogleClient();
            $client->setAuthConfig(json_decode($this->serviceAccount, true));
            $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
            $client->fetchAccessTokenWithAssertion();

            $accessToken = $client->getAccessToken();
            $accessTokenString = $accessToken['access_token'];

            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

            // Format data as strings for FCM V1 (limitations)
            $formattedData = [];
            foreach ($data as $key => $value) {
                $formattedData[$key] = (string) $value;
            }

            $payload = [
                'message' => [
                    'token' => $token,
                    'notification' => [
                        'title' => $title,
                        'body' => $body,
                    ],
                    'data' => (object) $formattedData, // Ensure empty array becomes object or valid json
                ],
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessTokenString,
                'Content-Type' => 'application/json',
            ])->post($url, $payload);

            if ($response->successful()) {
                Log::info("FCM V1 Sent: " . $response->body());
                return true;
            } else {
                Log::error('FCM V1 Error: ' . $response->body());
                return false;
            }

        } catch (\Exception $e) {
            Log::error('FCM V1 Exception: ' . $e->getMessage());
            return false;
        }
    }

    protected function sendLegacy($token, $title, $body, $data)
    {
        $url = 'https://fcm.googleapis.com/fcm/send';

        $payload = [
            'to' => $token,
            'notification' => [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
            ],
            'data' => $data,
        ];

        $response = Http::withHeaders([
            'Authorization' => 'key=' . $this->serverKey,
            'Content-Type' => 'application/json',
        ])->post($url, $payload);

        if ($response->successful()) {
            return true;
        } else {
            Log::error('FCM Legacy Error: ' . $response->body());
            return false;
        }
    }
}

