<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Dynamic Login Rate Limiter
        \Illuminate\Support\Facades\RateLimiter::for('login', function (\Illuminate\Http\Request $request) {
            $settings = \App\Models\GeneralSetting::firstOrNew();
            $maxAttempts = $settings->login_max_attempts ?? 5;
            $decayMinutes = $settings->login_decay_minutes ?? 1;

            return \Illuminate\Cache\RateLimiting\Limit::perMinutes($decayMinutes, $maxAttempts)->by($request->email . $request->ip());
        });

        // Share categories with the shop layout for mega menu
        view()->composer('layouts.shop', function ($view) {
            $view->with('menuCategories', \App\Models\Category::whereNull('parent_id')
                ->where('status', true)
                ->where(function ($query) {
                    // Has direct products
                    $query->whereHas('products', function ($q) {
                        $q->where('status', true);
                    })
                        // OR has child categories with products
                        ->orWhereHas('children', function ($q) {
                        $q->where('status', true)
                            ->whereHas('subProducts', function ($sq) {
                                $sq->where('status', true);
                            });
                    });
                })
                ->with([
                    'children' => function ($q) {
                        $q->where('status', true)
                            ->whereHas('subProducts', function ($sq) {
                                $sq->where('status', true);
                            });
                    }
                ])
                ->get());

            // Inject Custom Offers
            $offerService = new \App\Services\CustomOfferService();
            $view->with('customOffers', $offerService->getOffersForCurrentPage());

            $view->with('settings', \App\Models\GeneralSetting::firstOrNew());
        });

        // Share counts with admin sidebar
        view()->composer('admin.partials.sidebar', function ($view) {
            $view->with('pendingOrdersCount', \App\Models\Order::where('status', 'pending')->count());
            // Assuming User model has Notifiable trait
            $unreadCount = 0;
            if (\Illuminate\Support\Facades\Auth::check()) {
                $unreadCount = \Illuminate\Support\Facades\Auth::user()->unreadNotifications()->count();
            }
            $view->with('unreadNotificationsCount', $unreadCount);
        });

        // Dynamic Mail Configuration from Database
        try {
            $settings = \App\Models\GeneralSetting::first();
            if ($settings) {
                // Global From Address
                if ($settings->mail_from_address) {
                    config(['mail.from.address' => $settings->mail_from_address]);
                }
                if ($settings->mail_from_name) {
                    config(['mail.from.name' => $settings->mail_from_name]);
                }

                // Mailer Driver
                if ($settings->mail_mailer) {
                    config(['mail.default' => $settings->mail_mailer]);
                }

                // SMTP Configuration
                if ($settings->mail_mailer === 'smtp') {
                    config([
                        'mail.mailers.smtp.host' => $settings->mail_host,
                        'mail.mailers.smtp.port' => $settings->mail_port,
                        'mail.mailers.smtp.username' => $settings->mail_username,
                        'mail.mailers.smtp.password' => $settings->mail_password,
                        'mail.mailers.smtp.encryption' => $settings->mail_encryption,
                    ]);
                }

                // Mailgun Configuration
                if ($settings->mail_mailer === 'mailgun') {
                    config([
                        'services.mailgun.domain' => $settings->mailgun_domain,
                        'services.mailgun.secret' => $settings->mailgun_secret,
                        'services.mailgun.endpoint' => $settings->mailgun_endpoint,
                    ]);
                }
            }
        } catch (\Exception $e) {
            // Quietly fail if table not found or DB issue (runs during migration too)
        }
    }
}
