<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TicketStatusUpdated extends Notification
{
    use Queueable;

    public $ticket;
    public $action;

    /**
     * Create a new notification instance.
     */
    public function __construct($ticket, $action)
    {
        $this->ticket = $ticket;
        $this->action = $action;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', \App\Notifications\Channels\FCMChannel::class];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $message = $this->getMessage($notifiable);
        $url = $this->getUrl($notifiable);

        return [
            'ticket_id' => $this->ticket->id,
            'ticket_number' => $this->ticket->ticket_number,
            'action' => $this->action,
            'message' => $message,
            'url' => $url,
        ];
    }

    public function toFcm($notifiable)
    {
        $message = $this->getMessage($notifiable);
        $badgeCount = $notifiable->unreadNotifications()->count() + 1;

        return [
            'title' => 'Support Ticket Update',
            'body' => $message,
            'badge' => $badgeCount,
            'data' => [
                'ticket_id' => (string) $this->ticket->id,
                'ticket_number' => (string) $this->ticket->ticket_number,
                'type' => 'ticket_update',
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                'route' => 'ticket_detail',
                'badge' => (string) $badgeCount
            ]
        ];
    }

    protected function getMessage($notifiable)
    {
        switch ($this->action) {
            case 'created':
                return 'New ticket #' . $this->ticket->ticket_number . ' created by ' . $this->ticket->user->name;
            case 'replied_by_admin':
                return 'New response on your ticket #' . $this->ticket->ticket_number;
            case 'replied_by_user':
                return 'New reply from ' . $this->ticket->user->name . ' on ticket #' . $this->ticket->ticket_number;
            case 'closed':
                return 'Ticket #' . $this->ticket->ticket_number . ' has been closed.';
            default:
                return 'Update on ticket #' . $this->ticket->ticket_number;
        }
    }

    protected function getUrl($notifiable)
    {
        // Simple logic for web notifications, might need adjustment for admin/user portal separation
        return '#';
    }
}
