<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Order;
use App\Notifications\Channels\FCMChannel;

class CustomerOrderStatusNotification extends Notification
{
    use Queueable;

    protected $order;

    /**
     * Create a new notification instance.
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', FCMChannel::class];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'title' => 'Order Update',
            'message' => $this->getMessage(),
            'order_id' => $this->order->id,
            'status' => $this->order->status,
            'created_at' => now(),
        ];
    }

    public function toFcm($notifiable)
    {
        // Calculate unread notification count
        $badgeCount = $notifiable->unreadNotifications()->count() + 1; // +1 for this new one

        return [
            'title' => 'Order Update',
            'body' => $this->getMessage(),
            'badge' => $badgeCount,
            'data' => [
                'order_id' => $this->order->id,
                'type' => 'order_status_update',
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                'route' => 'orders',
                'badge' => (string) $badgeCount
            ]
        ];
    }

    protected function getMessage()
    {
        $status = $this->order->status;

        if ($status == 'pending') {
            return "Your order #{$this->order->order_number} has been successfully placed.";
        }

        $prettyStatus = ucfirst(str_replace('_', ' ', $status));
        return "Your order #{$this->order->order_number} is now {$prettyStatus}.";
    }
}
