<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, \Laravel\Sanctum\HasApiTokens;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'mobile',
        'password',
        'profile_photo_path',
        'is_active',
        'gender',
        'dob',
        'whatsapp_number',
        'is_deactivated',
        'wallet_balance',
        'fcm_token',
        'google_id',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected $appends = ['avatar_url'];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
        ];
    }

    public function getAvatarUrlAttribute()
    {
        if ($this->profile_photo_path) {
            return asset('storage/' . $this->profile_photo_path);
        }

        // Return a default placeholder if no image is set using ui-avatars.com for a nice generated initial avatar
        return 'https://ui-avatars.com/api/?name=' . urlencode($this->name) . '&color=7F9CF5&background=EBF4FF';
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class);
    }

    public function hasRole($role)
    {
        if (is_string($role)) {
            return $this->roles->contains('name', $role);
        }
        return !!$role->intersect($this->roles)->count();
    }

    public function hasPermission($permission)
    {
        return $this->roles->flatMap->permissions->contains('slug', $permission);
    }

    public function addresses()
    {
        return $this->hasMany(Address::class);
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function wishlist()
    {
        return $this->hasMany(Wishlist::class);
    }

    public function tickets()
    {
        return $this->hasMany(Ticket::class);
    }

    public function walletTransactions()
    {
        return $this->hasMany(WalletTransaction::class)->latest();
    }

    public function creditWallet($amount, $description = null, $refType = null, $refId = null)
    {
        $this->increment('wallet_balance', $amount);
        $this->walletTransactions()->create([
            'type' => 'credit',
            'amount' => $amount,
            'description' => $description,
            'reference_type' => $refType,
            'reference_id' => $refId
        ]);
    }

    public function debitWallet($amount, $description = null, $refType = null, $refId = null)
    {
        if ($this->wallet_balance < $amount) {
            throw new \Exception("Insufficient wallet balance");
        }
        $this->decrement('wallet_balance', $amount);
        $this->walletTransactions()->create([
            'type' => 'debit',
            'amount' => $amount,
            'description' => $description,
            'reference_type' => $refType,
            'reference_id' => $refId
        ]);
    }
    public function createdGiftCards()
    {
        return $this->hasMany(GiftCard::class, 'created_by');
    }

    public function redeemedGiftCards()
    {
        return $this->hasMany(GiftCard::class, 'redeemed_by');
    }

    public function paymentMethods()
    {
        return $this->hasMany(UserPaymentMethod::class);
    }
}
