<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'is_returnable' => 'boolean',
        'return_window_days' => 'integer',
    ];

    protected $appends = ['thumbnail_url'];

    public function getThumbnailUrlAttribute()
    {
        $image = $this->images->first();

        if (!$image) {
            return asset('images/placeholder.png');
        }

        // Determine which path to use (prefer thumbnail, fall back to main image)
        $path = !empty($image->thumbnail_path) ? $image->thumbnail_path : $image->image_path;

        // If path is already a full URL (e.g., S3), return it directly
        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        // Otherwise, return full URL using asset helper
        // Ensure path doesn't start with / to avoid double slashes with asset() which handles base URL
        $path = ltrim($path, '/');

        return asset($path);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function subcategory()
    {
        return $this->belongsTo(Category::class, 'subcategory_id');
    }

    public function images()
    {
        return $this->hasMany(ProductImage::class)->orderBy('sort_order');
    }

    public function variations()
    {
        return $this->hasMany(ProductVariation::class)->orderBy('sort_order');
    }

    public function upsells()
    {
        return $this->belongsToMany(Product::class, 'product_relations', 'product_id', 'related_product_id')
            ->wherePivot('type', 'upsell')
            ->withTimestamps();
    }

    public function crossSells()
    {
        return $this->belongsToMany(Product::class, 'product_relations', 'product_id', 'related_product_id')
            ->wherePivot('type', 'cross_sell')
            ->withTimestamps();
    }

    public function discounts()
    {
        return $this->morphToMany(Discount::class, 'discountable');
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function reviews()
    {
        return $this->hasMany(ProductReview::class);
    }

    public function publishedReviews()
    {
        return $this->hasMany(ProductReview::class)->where('is_published', true);
    }
}
