<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    protected $guarded = [];

    protected $casts = [
        'created_at' => 'datetime',
    ];

    protected static function booted()
    {
        static::created(function ($order) {
            if ($order->status && $order->status !== 'pending_payment') {
                $order->statusHistories()->create(['status' => $order->status]);

                // Notify Admins
                $admins = \App\Models\User::whereHas('roles', function ($q) {
                    $q->where('name', 'Admin');
                })->get();

                \Illuminate\Support\Facades\Notification::send($admins, new \App\Notifications\AdminOrderNotification($order));
            }
        });

        static::updated(function ($order) {
            if ($order->isDirty('status')) {
                // If moving away from pending_payment, it counts as "Placed"
                $order->statusHistories()->create(['status' => $order->status]);

                if ($order->getOriginal('status') === 'pending_payment' && $order->status !== 'pending_payment') {
                    // Notify Admins
                    $admins = \App\Models\User::whereHas('roles', function ($q) {
                        $q->where('name', 'Admin');
                    })->get();

                    \Illuminate\Support\Facades\Notification::send($admins, new \App\Notifications\AdminOrderNotification($order));
                }

                // Notify Customer
                if ($order->user && $order->user->fcm_token) {
                    $order->user->notify(new \App\Notifications\CustomerOrderStatusNotification($order));
                }
            }

            if ($order->isDirty('status') && $order->status === 'completed') {
                $orderScope = $order->order_scope; // Accessor
                if ($orderScope === 'Local' && $order->delivery_agent_id) {
                    $settings = \App\Models\GeneralSetting::first();
                    if ($settings && $settings->delivery_wallet_enabled && $settings->delivery_wallet_earn_rate > 0) {
                        $points = round(($order->total / 100) * $settings->delivery_wallet_earn_rate);
                        if ($points > 0) {
                            $agent = $order->deliveryAgent()->with('user')->first();
                            if ($agent && $agent->user) {
                                $agent->user->creditWallet($points, "Delivery Reward for Order #{$order->order_number}", 'order', $order->id);
                            }
                        }
                    }
                }
            }

            if ($order->isDirty('delivery_agent_id') && $order->delivery_agent_id) {
                // Notify the agent
                $agent = $order->deliveryAgent()->with('user')->first();
                if ($agent && $agent->user) {
                    $agent->user->notify(new \App\Notifications\DeliveryAgentNotification(
                        "You have been assigned to Order #" . $order->order_number,
                        "New Order Assigned"
                    ));
                }
            }
        });
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function deliveryAgent()
    {
        return $this->belongsTo(DeliveryAgent::class);
    }

    public function shippingAddress()
    {
        return $this->belongsTo(Address::class, 'shipping_address_id');
    }

    public function billingAddress()
    {
        return $this->belongsTo(Address::class, 'billing_address_id');
    }

    public function getOrderScopeAttribute()
    {
        if (!$this->shippingAddress) {
            return 'Unknown';
        }

        // Default to first active store for comparison, or hardcoded India/Delhi if no store found
        $store = \App\Models\Store::where('is_active', true)->first();

        $storeCountry = $store ? $store->country : 'India';
        $storeState = $store ? $store->state : ''; // fallback needed if no store

        // Normalize strings for comparison
        $shippingCountry = strtolower(trim($this->shippingAddress->country));
        $shippingState = strtolower(trim($this->shippingAddress->state ?? ''));
        $storeCountry = strtolower(trim($storeCountry));
        $storeState = strtolower(trim($storeState));

        if ($shippingCountry !== $storeCountry) {
            return 'International';
        }

        if ($storeState && $shippingState !== $storeState) {
            return 'National';
        }

        return 'Local';
    }
    public function pickupStore()
    {
        return $this->belongsTo(Store::class, 'pickup_store_id');
    }

    public function statusHistories()
    {
        return $this->hasMany(OrderStatusHistory::class)->oldest();
    }

    public function productReviews()
    {
        return $this->hasMany(ProductReview::class);
    }

    public function agentRating()
    {
        return $this->hasOne(AgentRating::class);
    }
    public function returnRequest()
    {
        return $this->hasOne(OrderReturn::class);
    }
}
