<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomOffer extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'layout_id',
        'status',
        'placement_pages',
        'target_element',
        'placement_position',
        'border_radius',
        'box_shadow',
        'bg_color',
        'heading_font_size',
        'heading_font_weight',
        'heading_color',
    ];
    protected $casts = [
        'placement_pages' => 'array',
        'status' => 'boolean',
    ];

    public function layout()
    {
        return $this->belongsTo(Layout::class);
    }

    public function items()
    {
        return $this->hasMany(CustomOfferItem::class);
    }

    /**
     * Get the preview URL based on the first selected placement.
     */
    public function getPreviewUrlAttribute()
    {
        $pages = $this->placement_pages ?? [];
        
        if (empty($pages)) {
            return route('home');
        }

        // Just take the first one to preview
        $target = $pages[0];

        if ($target === 'home') {
            return route('home');
        }

        if ($target === 'shop') {
            return route('shop.index');
        }

        if (str_starts_with($target, 'page-')) {
            $pageId = str_replace('page-', '', $target);
            $page = \App\Models\Page::find($pageId);
            return $page ? route('pages.show', $page->slug) : route('home');
        }
        
        // Category format in select: cat-{id}
        // But options logic in controller stored slug? 
        // Let's re-verify Controller logic. 
        // Controller: 'slug' => 'collections/' ... 
        // But the value of the option? 
        // options->push((object) [ 'id' => 'cat-' . $subcategory->id ... ])
        // Controller logic for getPlacementOptions:
        // $options->push((object) [ 'id' => 'cat-' . $subcategory->id, 'title' => ..., 'slug' => ... ])
        // Wait, the select logic in create/edit uses `id` as key if we iterate options.
        // Let's check view.
        
        // Actually, let's look at `create.blade.php`.
        // x-data: placementPages: ...
        // The value stored is the ID (e.g., 'home', 'shop', 'page-1', 'cat-5').
        // So we need to resolve that ID to a URL.
        
        if (str_starts_with($target, 'cat-')) {
            $catId = str_replace('cat-', '', $target);
            $category = \App\Models\Category::with('parent')->find($catId);
            
            if ($category && $category->parent) {
                return route('shop.subcategory', [
                    'category' => $category->parent->slug, 
                    'subcategory' => $category->slug
                ]);
            }
        }

        return route('home');
    }
}
