<?php

namespace App\Imports;

use App\Models\Product;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Row;
use Illuminate\Support\Str;
use App\Models\Category;
use App\Models\ProductVariation;

class ProductsImport implements OnEachRow, WithHeadingRow, WithValidation
{
    public function rules(): array
    {
        return [
            'product_name' => 'required',
            'category' => 'required',
            'mrp' => 'required|numeric',
            'selling_price_sp' => 'required|numeric',
            'short_description' => 'required',
            'dispatch_days' => 'required|integer',
        ];
    }

    public function onRow(Row $row)
    {
        $rowIndex = $row->getIndex();
        $row = $row->toArray();

        // Ensure mandatory category exists or create it
        $category = Category::where('name', $row['category'])->first();
        if (!$category) {
            $category = Category::create([
                'name' => $row['category'],
                'slug' => Str::slug($row['category']),
                'status' => true
            ]);
        }

        $subcategory = null;
        if (!empty($row['sub_category'])) {
            $subcategory = Category::where('name', $row['sub_category'])
                ->where('parent_id', $category->id)
                ->first();

            if (!$subcategory) {
                $subcategory = Category::create([
                    'name' => $row['sub_category'],
                    'slug' => Str::slug($row['sub_category']),
                    'parent_id' => $category->id,
                    'status' => true
                ]);
            }
        }

        $stock = $row['stock_quantity'] ?? 100;

        $product = Product::create([
            'name' => $row['product_name'],
            'slug' => Str::slug($row['product_name']) . '-' . rand(1000, 9999),
            'category_id' => $category->id,
            'subcategory_id' => $subcategory ? $subcategory->id : null,
            'mrp' => $row['mrp'],
            'sp' => $row['selling_price_sp'],
            'reseller_price' => $row['reseller_price'] ?? $row['selling_price_sp'],
            'number_of_units' => $row['number_of_units'] ?? 1,
            'short_description' => $row['short_description'],
            'long_description' => $row['short_description'],
            'dispatch_days' => $row['dispatch_days'],
            'tax_code' => '18',
            'status' => true,
        ]);

        // Create Default Variation
        ProductVariation::create([
            'product_id' => $product->id,
            'mrp' => $row['mrp'],
            'sp' => $row['selling_price_sp'],
            'reseller_price' => $row['reseller_price'] ?? $row['selling_price_sp'],
            'stock' => $row['stock_quantity'] ?? 100, // Default to 100 if not provided
            'sku' => strtoupper(Str::slug($row['product_name'])),
        ]);
    }
}
