<?php

namespace App\Http\Middleware;

use App\Models\TrafficHistory;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Session;

class TrackTrafficSource
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Track only if it's a GET request and not an AJAX call
        if ($request->isMethod('GET') && !$request->ajax() && !$request->is('admin/*') && !$request->is('api/*')) {
            $sessionId = Session::getId();

            // Check if we've already tracked this session in the last 24 hours to avoid spamming
            $alreadyTracked = TrafficHistory::where('session_id', $sessionId)
                ->where('created_at', '>=', now()->subHours(24))
                ->exists();

            if (!$alreadyTracked) {
                $referrer = $request->headers->get('referer');
                $source = $this->determineSource($referrer);
                $device = $this->determineDevice($request->userAgent());

                TrafficHistory::create([
                    'session_id' => $sessionId,
                    'ip_address' => $request->ip(),
                    'source' => $source,
                    'referrer' => $referrer,
                    'landing_page' => $request->fullUrl(),
                    'device' => $device,
                    'user_agent' => $request->userAgent(),
                ]);
            }
        }

        return $next($request);
    }

    private function determineSource($referrer)
    {
        if (empty($referrer)) {
            return 'Direct';
        }

        $host = parse_url($referrer, PHP_URL_HOST);

        if (str_contains($host, 'google.')) {
            return 'Google Search';
        }
        if (str_contains($host, 'bing.')) {
            return 'Bing Search';
        }
        if (str_contains($host, 'facebook.com') || str_contains($host, 'fb.me')) {
            return 'Facebook';
        }
        if (str_contains($host, 'instagram.com')) {
            return 'Instagram';
        }
        if (str_contains($host, 't.co') || str_contains($host, 'twitter.com') || str_contains($host, 'x.com')) {
            return 'Twitter/X';
        }
        if (str_contains($host, 'youtube.com')) {
            return 'YouTube';
        }
        if (str_contains($host, 'linkedin.com')) {
            return 'LinkedIn';
        }
        if (str_contains($host, 'pinterest.com')) {
            return 'Pinterest';
        }

        // Internal referral (if it's our own site, though unlikely for first landing)
        $ourHost = parse_url(config('app.url'), PHP_URL_HOST);
        if ($host === $ourHost) {
            return 'Direct';
        }

        return 'Referral (' . $host . ')';
    }

    private function determineDevice($userAgent)
    {
        $userAgent = strtolower($userAgent);
        if (str_contains($userAgent, 'mobile') || str_contains($userAgent, 'android') || str_contains($userAgent, 'iphone')) {
            return 'Mobile';
        }
        if (str_contains($userAgent, 'tablet') || str_contains($userAgent, 'ipad')) {
            return 'Tablet';
        }
        return 'Desktop';
    }
}
