<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Wishlist;
use App\Models\Product;
use Illuminate\Support\Facades\Auth;

class WishlistController extends Controller
{
    public function index()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Wishlist functionality is disabled for delivery agents.');
        }

        $user = Auth::user();
        if (!$user) {
            return redirect()->route('login');
        }

        $wishlists = $user->wishlist()->with(['product.images', 'product.variations'])->latest()->get();
        $settings = \App\Models\GeneralSetting::firstOrNew();

        return view('user.wishlist', compact('wishlists', 'settings'));
    }

    public function toggle(Request $request)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            if ($request->wantsJson()) {
                return response()->json(['status' => 'error', 'message' => 'Action not allowed for delivery agents.']);
            }
            return redirect()->route('delivery.dashboard')->with('error', 'Wishlist functionality is disabled for delivery agents.');
        }

        if (!Auth::check()) {
            return response()->json(['status' => 'error', 'message' => 'Please login first', 'redirect' => route('login')], 401);
        }

        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $user = Auth::user();
        $productId = $request->product_id;

        $wishlist = Wishlist::where('user_id', $user->id)
            ->where('product_id', $productId)
            ->first();

        if ($wishlist) {
            $wishlist->delete();
            $status = 'removed';
            $message = 'Item removed from wishlist';
        } else {
            Wishlist::create([
                'user_id' => $user->id,
                'product_id' => $productId
            ]);
            $status = 'added';
            $message = 'Item added to wishlist';
        }

        if ($request->wantsJson()) {
            return response()->json([
                'status' => $status,
                'message' => $message,
                'count' => $user->wishlist()->count()
            ]);
        }

        return back()->with('success', $message);
    }

    public function remove($id)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Wishlist functionality is disabled for delivery agents.');
        }

        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $wishlist = Wishlist::where('user_id', Auth::id())->where('id', $id)->first();
        if ($wishlist) {
            $wishlist->delete();
            return back()->with('success', 'Item removed from wishlist');
        }

        return back()->with('error', 'Item not found in wishlist');
    }

    public function moveToCart($id)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Wishlist functionality is disabled for delivery agents.');
        }

        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $wishlist = Wishlist::where('user_id', Auth::id())->with('product.variations')->where('id', $id)->first();

        if (!$wishlist) {
            return back()->with('error', 'Item not found in wishlist');
        }

        $product = $wishlist->product;

        // Determine product details (variation/stock)
        // Default to first variation or main product if simple
        $variationId = null;
        $price = $product->sp;
        $stock = 0;

        if ($product->variations->isNotEmpty()) {
            // Logic: Pick the first available variation? Or redirect to product page?
            // "Move to cart" implies direct action. Let's try to pick the first in-stock variation or just the first one.
            $variation = $product->variations->where('stock', '>', 0)->first();
            if (!$variation) {
                $variation = $product->variations->first(); // Fallback even if OOS
            }

            if ($variation) {
                $variationId = $variation->id;
                $price = $variation->sp;
                $stock = $variation->stock;
            }
        } else {
            // Simple product logic if applicable (though structure suggests variations are primary)
            // Assuming product entity might track stock if no variations or purely based on variations.
            // Previous code suggests variations imply stock.
            $stock = 0; // If no variations, assume OOS or logic from cart controller
        }

        if ($stock <= 0 && $product->variations->isNotEmpty()) {
            return back()->with('error', 'Product is out of stock.');
        }

        // Get/Create Cart
        $cart = \App\Models\Cart::firstOrCreate(['user_id' => Auth::id()]);

        // Add to Cart
        $existingItem = $cart->items()
            ->where('product_id', $product->id)
            ->where('product_variation_id', $variationId)
            ->first();

        if ($existingItem) {
            $existingItem->quantity += 1;
            $existingItem->save();
        } else {
            $cart->items()->create([
                'product_id' => $product->id,
                'product_variation_id' => $variationId,
                'quantity' => 1,
                'price' => $price,
            ]);
        }

        // Remove from Wishlist
        $wishlist->delete();

        return redirect()->route('cart.index')->with('success', 'Item moved to cart!');
    }
}
