<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class UserTicketController extends Controller
{
    public function index(Request $request)
    {
        if (\Illuminate\Support\Facades\Auth::check() && \Illuminate\Support\Facades\Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $status = $request->get('status', 'open');
        $tickets = \Illuminate\Support\Facades\Auth::user()->tickets()
            ->with('order')
            ->where('status', $status)
            ->latest()
            ->paginate(10);

        $counts = [
            'open' => \Illuminate\Support\Facades\Auth::user()->tickets()->where('status', 'open')->count(),
            'closed' => \Illuminate\Support\Facades\Auth::user()->tickets()->where('status', 'closed')->count(),
        ];

        return view('user.tickets.index', compact('tickets', 'status', 'counts'));
    }

    public function create(\App\Models\Order $order)
    {
        if (\Illuminate\Support\Facades\Auth::check() && \Illuminate\Support\Facades\Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        // Check if order belongs to user
        if ($order->user_id !== \Illuminate\Support\Facades\Auth::id()) {
            abort(403);
        }
        return view('user.tickets.create', compact('order'));
    }

    public function store(Request $request, \App\Models\Order $order)
    {
        if ($order->user_id !== \Illuminate\Support\Facades\Auth::id()) {
            abort(403);
        }

        $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'attachment' => 'nullable|file|max:2048|mimes:jpg,jpeg,png,pdf,doc,docx',
        ]);

        $ticket = \App\Models\Ticket::create([
            'user_id' => \Illuminate\Support\Facades\Auth::id(),
            'order_id' => $order->id,
            'ticket_number' => 'TKT-' . strtoupper(\Illuminate\Support\Str::random(8)),
            'subject' => $request->subject,
            'status' => 'open',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('tickets', 'public');
        }

        $ticket->messages()->create([
            'user_id' => \Illuminate\Support\Facades\Auth::id(),
            'message' => $request->message,
            'attachment_path' => $attachmentPath,
        ]);

        // Notify Admin
        \App\Models\User::whereHas('roles', function ($q) {
            $q->where('name', 'admin');
        })->get()->each(function ($admin) use ($ticket) {
            $admin->notify(new \App\Notifications\TicketStatusUpdated($ticket, 'created'));
        });

        return redirect()->route('user.tickets.show', $ticket)->with('success', 'Ticket created successfully.');
    }

    public function show(\App\Models\Ticket $ticket)
    {
        if ($ticket->user_id !== \Illuminate\Support\Facades\Auth::id()) {
            abort(403);
        }

        // Mark notifications as read
        \Illuminate\Support\Facades\Auth::user()->unreadNotifications()
            ->where('data->ticket_id', $ticket->id)
            ->get()->markAsRead();

        $ticket->load(['messages.user', 'order']);
        return view('user.tickets.show', compact('ticket'));
    }

    public function reply(Request $request, \App\Models\Ticket $ticket)
    {
        if ($ticket->user_id !== \Illuminate\Support\Facades\Auth::id()) {
            abort(403);
        }

        if ($ticket->status === 'closed') {
            return back()->with('error', 'Ticket is closed.');
        }

        $request->validate([
            'message' => 'required|string',
            'attachment' => 'nullable|file|max:2048',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('tickets', 'public');
        }

        $ticket->messages()->create([
            'user_id' => \Illuminate\Support\Facades\Auth::id(),
            'message' => $request->message,
            'attachment_path' => $attachmentPath,
        ]);

        // Notify Admin
        \App\Models\User::whereHas('roles', function ($q) {
            $q->where('name', 'admin');
        })->get()->each(function ($admin) use ($ticket) {
            $admin->notify(new \App\Notifications\TicketStatusUpdated($ticket, 'replied_by_user'));
        });

        return back()->with('success', 'Reply sent successfully.');
    }

    public function close(\App\Models\Ticket $ticket)
    {
        if ($ticket->user_id !== \Illuminate\Support\Facades\Auth::id()) {
            abort(403);
        }

        $ticket->update(['status' => 'closed']);
        return back()->with('success', 'Ticket closed successfully.');
    }
}
