<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class UserGiftCardController extends Controller
{
    public function index()
    {
        if (auth()->check() && auth()->user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $user = auth()->user();
        $giftCards = $user->createdGiftCards()->latest()->get();

        $receivedGiftCards = \App\Models\GiftCard::where('recipient_email', $user->email)
            ->latest()
            ->get();

        return view('user.gift-cards.index', compact('giftCards', 'receivedGiftCards'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'recipient_email' => 'nullable|email',
        ]);

        $user = auth()->user();
        $amount = $request->amount;
        $settings = \App\Models\GeneralSetting::first();

        if (!$settings->gift_card_enabled) {
            return back()->with('error', 'Gift card system is currently disabled.');
        }

        if ($user->wallet_balance < $amount) {
            return back()->with('error', 'Insufficient wallet balance.');
        }

        try {
            // Generate Code first to use ID if needed (but we use random string)
            $code = strtoupper('GC-' . \Illuminate\Support\Str::random(12));
            $expiryDays = $settings->gift_card_expiry_days ?? 365;
            $expiresAt = now()->addDays($expiryDays);

            // Create Gift Card first so we have ID for transaction reference
            $giftCard = \App\Models\GiftCard::create([
                'code' => $code,
                'amount' => $amount,
                'created_by' => $user->id,
                'recipient_email' => $request->recipient_email,
                'expires_at' => $expiresAt,
                'status' => 'active'
            ]);

            // Debit Wallet
            $user->debitWallet($amount, "Created Gift Card {$code}", 'gift_card', $giftCard->id);

            return back()->with('success', 'Gift card created successfully!');

        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
}
