<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Cart;
use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\Order;

class UserController extends Controller
{
    // Dashboard Overview
    public function dashboard()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $user = Auth::user();

        $totalOrders = $user->orders()->count();
        $pendingOrders = $user->orders()->whereIn('status', ['pending', 'processing'])->count();
        $wishlistCount = \App\Models\Wishlist::where('user_id', $user->id)->count();
        $recentOrders = $user->orders()->where('status', '!=', 'pending_payment')->latest()->take(5)->get();

        return view('user.dashboard', compact('user', 'totalOrders', 'pendingOrders', 'wishlistCount', 'recentOrders'));
    }

    // Profile Management
    public function profile()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }
        $user = Auth::user();
        return view('user.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'whatsapp_number' => 'nullable|string|max:20',
            'gender' => 'nullable|in:male,female,other',
            'dob' => 'nullable|date',
            'profile_photo' => 'nullable|image|max:2048',
        ]);

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($user->profile_photo_path);
            }
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $user->profile_photo_path = $path;
        }

        $user->update([
            'name' => $request->name,
            'whatsapp_number' => $request->whatsapp_number,
            'gender' => $request->gender,
            'dob' => $request->dob,
        ]);

        return back()->with('success', 'Profile updated successfully.');
    }

    // Address Management
    public function addresses()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $addresses = Auth::user()->addresses;
        return view('user.addresses', compact('addresses'));
    }

    public function storeAddress(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'phone' => 'required|string',
            'postal_code' => 'required|string',
            'address_line1' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'country' => 'required|string',
            'type' => 'required|in:home,office,other',
        ]);

        $isDefault = $request->has('is_default') || Auth::user()->addresses()->count() === 0;

        if ($isDefault) {
            Auth::user()->addresses()->update(['is_default' => false]);
        }

        Auth::user()->addresses()->create([
            'name' => $request->name,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'address_line1' => $request->address_line1,
            'address_line2' => $request->address_line2,
            'city' => $request->city,
            'state' => $request->state,
            'country' => $request->country,
            'type' => $request->type,
            'open_time' => $request->type == 'office' ? $request->open_time : null,
            'close_time' => $request->type == 'office' ? $request->close_time : null,
            'is_default' => $isDefault,
        ]);

        return back()->with('success', 'Address added successfully.');
    }

    public function updateAddress(Request $request, \App\Models\Address $address)
    {
        // Ensure user owns the address
        if ($address->user_id !== Auth::id()) {
            return back()->with('error', 'Unauthorized action.');
        }

        $request->validate([
            'name' => 'required|string',
            'phone' => 'required|string',
            'postal_code' => 'required|string',
            'address_line1' => 'required|string',
            'city' => 'required|string',
            'state' => 'required|string',
            'country' => 'required|string',
            'type' => 'required|in:home,office,other',
        ]);

        $isDefault = $request->has('is_default');

        if ($isDefault) {
            // If setting as default, unset others
            Auth::user()->addresses()->where('id', '!=', $address->id)->update(['is_default' => false]);
            $address->is_default = true;
        } else {
            $address->is_default = false;
        }

        $address->update([
            'name' => $request->name,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'address_line1' => $request->address_line1,
            'address_line2' => $request->address_line2,
            'city' => $request->city,
            'state' => $request->state,
            'country' => $request->country,
            'type' => $request->type,
            'open_time' => $request->type == 'office' ? $request->open_time : null,
            'close_time' => $request->type == 'office' ? $request->close_time : null,
            'is_default' => $address->is_default,
        ]);

        return back()->with('success', 'Address updated successfully.');
    }

    public function destroyAddress(\App\Models\Address $address)
    {
        // Ensure user owns the address
        if ($address->user_id !== Auth::id()) {
            return back()->with('error', 'Unauthorized action.');
        }

        $address->delete();

        return back()->with('success', 'Address deleted successfully.');
    }

    // Order Management
    public function orders(Request $request)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $status = $request->get('status', 'all');
        $query = Auth::user()->orders()
            ->where('status', '!=', 'pending_payment');

        if ($status !== 'all') {
            if ($status === 'returned') {
                $query->whereHas('returnRequest');
            } else {
                $query->where('status', $status);
            }
        }

        if ($request->start_date) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->end_date) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $orders = $query->with(['agentRating', 'productReviews', 'returnRequest'])->latest()->get();

        // Calculate counts
        $counts = [
            'all' => Auth::user()->orders()->where('status', '!=', 'pending_payment')->count(),
            'pending' => Auth::user()->orders()->where('status', 'pending')->count(),
            'processing' => Auth::user()->orders()->where('status', 'processing')->count(),
            'shipped' => Auth::user()->orders()->where('status', 'shipped')->count(),
            'completed' => Auth::user()->orders()->where('status', 'completed')->count(),
            'cancelled' => Auth::user()->orders()->where('status', 'cancelled')->count(),
            'returned' => Auth::user()->orders()->whereHas('returnRequest')->count(),
        ];

        return view('user.orders', compact('orders', 'status', 'counts'));
    }

    public function showOrder(\App\Models\Order $order)
    {
        // Ensure user owns the order
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress', 'statusHistories', 'deliveryAgent.user', 'productReviews', 'agentRating']);

        return view('user.orders.show', compact('order'));
    }

    public function getOrderLocation(\App\Models\Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $agent = $order->deliveryAgent;
        if (!$agent) {
            return response()->json(['error' => 'No agent assigned'], 404);
        }

        return response()->json([
            'lat' => $agent->current_latitude,
            'lng' => $agent->current_longitude,
            'status' => $order->status,
            'updated_at' => $agent->updated_at->toDateTimeString(),
        ]);
    }

    public function downloadInvoice(\App\Models\Order $order)
    {
        // Ensure user owns the order
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress']);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.invoice', compact('order'));
        return $pdf->download('invoice-' . $order->order_number . '.pdf');
    }

    public function wishlist()
    {
        // Placeholder: Fetch wishlist items (empty for now)
        $wishlistItems = collect([]);
        return view('user.wishlist', compact('wishlistItems'));
    }

    public function support()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }
        return view('user.support');
    }

    // Settings
    public function settings()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $user = Auth::user();
        return view('user.settings', compact('user'));
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = Auth::user();

        if (!\Illuminate\Support\Facades\Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password does not match.']);
        }

        $user->update([
            'password' => \Illuminate\Support\Facades\Hash::make($request->password)
        ]);

        return back()->with('success', 'Password updated successfully.');
    }

    public function deactivateAccount(Request $request)
    {
        $user = Auth::user();
        $user->update(['is_deactivated' => true, 'is_active' => false]);
        Auth::logout();
        return redirect()->route('home')->with('success', 'Your account has been deactivated.');
    }
    public function wallet()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard');
        }

        $user = Auth::user();
        $transactions = $user->walletTransactions()->paginate(10);
        return view('user.wallet', compact('user', 'transactions'));
    }

    public function reorder(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $cart = Cart::firstOrCreate(['user_id' => Auth::id()]);
        $messages = [];
        $addedCount = 0;

        foreach ($order->items as $item) {
            $product = $item->product;
            // Check if product exists and is active (using model default scope or status check)
            if (!$product || ($product->status == 0)) {
                $messages[] = "Product '" . ($product ? $product->name : 'Unknown') . "' is no longer available.";
                continue;
            }

            $variationId = null;
            if (isset($item->options['variation_id'])) {
                $variationId = $item->options['variation_id'];
            }

            $quantity = $item->quantity;
            $price = $product->sp;
            $variation = null;

            if ($variationId) {
                $variation = ProductVariation::find($variationId);

                if (!$variation) {
                    $messages[] = "Variation for '" . $product->name . "' is no longer available.";
                    continue;
                }
                $price = $variation->sp;

                if ($variation->stock < $quantity) {
                    if ($variation->stock > 0) {
                        $quantity = $variation->stock;
                        $messages[] = "Added only {$quantity} of '" . $product->name . " - " . ($item->options['attributes'][0]['value'] ?? 'Variation') . "' due to stock limits.";
                    } else {
                        $messages[] = "'" . $product->name . " - " . ($item->options['attributes'][0]['value'] ?? 'Variation') . "' is out of stock.";
                        continue;
                    }
                }
            }
            // If no variation, we assume no stock tracking or unlimited for now as per schema analysis.

            // Add to cart
            $existing = $cart->items()
                ->where('product_id', $product->id)
                ->where('product_variation_id', $variationId)
                ->first();

            if ($existing) {
                $existing->quantity += $quantity;
                // Re-check stock for accumulation?
                if ($variation && $existing->quantity > $variation->stock) {
                    $existing->quantity = $variation->stock;
                    // Message might be redundant if we just reduced it above, but safe.
                }
                $existing->save();
            } else {
                $cart->items()->create([
                    'product_id' => $product->id,
                    'product_variation_id' => $variationId,
                    'quantity' => $quantity,
                    'price' => $price
                ]);
            }
            $addedCount++;
        }

        if ($addedCount > 0) {
            if (count($messages) > 0) {
                return redirect()->route('cart.index')->with('success', 'Order items added to cart with some adjustments: ' . implode(' ', $messages));
            }
            return redirect()->route('cart.index')->with('success', 'Order items added to cart successfully.');
        } else {
            $errorMsg = count($messages) > 0 ? implode(' ', $messages) : 'No items could be reordered.';
            return back()->with('error', 'Could not reorder items. ' . $errorMsg);
        }
    }

    public function cancelOrder(Request $request, Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $request->validate([
            'cancellation_reason' => 'required|string|max:255',
        ]);

        // Allowed statuses for cancellation
        // "only cancel the order in pending, processed and completed status"
        // Implicitly, 'transit', 'shipped' (alias for transit usually), 'cancelled' are NOT allowed.
        // Assuming 'shipped' maps to 'transit' logic or vice versa. The user said 'transit'.
        // In this system, standard statuses are usually: pending, processing, shipped, delivered, completed, cancelled.
        // I will check against the ALLOWED list.

        $allowedStatuses = ['pending', 'processing', 'processed', 'completed'];

        if (!in_array($order->status, $allowedStatuses)) {
            // Specifically mentioning the rule user requested
            if ($order->status == 'transit' || $order->status == 'shipped') {
                return back()->with('error', 'Orders in transit cannot be cancelled.');
            }
            return back()->with('error', "Order cannot be cancelled in '{$order->status}' status.");
        }

        try {
            \Illuminate\Support\Facades\DB::transaction(function () use ($order, $request) {
                // Update Status
                $previousStatus = $order->status;
                $order->update([
                    'status' => 'cancelled',
                    'cancellation_reason' => $request->cancellation_reason,
                    'delivery_agent_id' => null, // Unassign Delivery Agent
                    // We do not auto-refund payment here as it might be complex (gateway refund).
                    // Admin should handle refund manually or specifically requested.
                ]);

                // Restore Stock
                foreach ($order->items as $item) {
                    if ($item->variation_id) {
                        $item->variation->increment('stock', $item->quantity);
                    } elseif ($item->product) {
                        // Assuming product might have simple stock if no variations,
                        // but schema emphasizes variations.
                        // If product schema has 'stock', increment it?
                        // Checking Product model usage generally suggests variations hold the stock.
                        // But let's check if Product has stock field just in case.
                        // (Safest is to try/catch or check existence, but based on reorder logic, we often check variations).
                        // I will skip if no variation, unless product has stock column.
                        // Actually, earlier code in Checkout decremented variation stock.
                        // Checkout also touched `ProductVariation::where('product_id')->first()` as fallback.
                        // I'll replicate that safety.
                        $variation = \App\Models\ProductVariation::where('product_id', $item->product_id)->first();
                        if ($variation) {
                            $variation->increment('stock', $item->quantity);
                        }
                    }
                }
            });

            return back()->with('success', 'Order has been cancelled successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel order: ' . $e->getMessage());
        }
    }
    public function submitRating(Request $request, Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        if ($order->status !== 'completed') {
            return back()->with('error', 'You can only rate completed orders.');
        }

        $request->validate([
            'agent_rating' => 'nullable|integer|between:1,5',
            'agent_comment' => 'nullable|string|max:1000',
            'product_ratings' => 'required|array',
            'product_ratings.*.id' => 'required|exists:products,id',
            'product_ratings.*.rating' => 'required|integer|between:1,5',
            'product_ratings.*.comment' => 'nullable|string|max:1000',
        ]);

        // Rate Agent
        if ($request->filled('agent_rating') && $order->delivery_agent_id) {
            \App\Models\AgentRating::updateOrCreate(
                ['order_id' => $order->id, 'delivery_agent_id' => $order->delivery_agent_id],
                [
                    'user_id' => Auth::id(),
                    'rating' => $request->agent_rating,
                    'comment' => $request->agent_comment
                ]
            );
        }

        // Rate Products
        foreach ($request->product_ratings as $prodRate) {
            \App\Models\ProductReview::updateOrCreate(
                ['order_id' => $order->id, 'product_id' => $prodRate['id']],
                [
                    'user_id' => Auth::id(),
                    'rating' => $prodRate['rating'],
                    'comment' => $prodRate['comment']
                ]
            );
        }

        return back()->with('success', 'Thank you for your feedback!');
    }
}
