<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\UserPaymentMethod;
use Illuminate\Support\Facades\Auth;

class PaymentMethodController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'type' => 'required|in:bank_transfer,upi',
            'account_number' => 'required_if:type,bank_transfer|nullable|string|max:255',
            'ifsc_code' => 'required_if:type,bank_transfer|nullable|string|max:255',
            'bank_name' => 'required_if:type,bank_transfer|nullable|string|max:255',
            'branch' => 'required_if:type,bank_transfer|nullable|string|max:255',
            'upi_id' => 'required_if:type,upi|nullable|string|max:255',
        ]);

        $user = Auth::user();

        // If it's the first method, make it default
        $isDefault = $user->paymentMethods()->count() === 0;

        $user->paymentMethods()->create([
            'type' => $request->type,
            'account_number' => $request->account_number,
            'ifsc_code' => $request->ifsc_code,
            'bank_name' => $request->bank_name,
            'branch' => $request->branch,
            'upi_id' => $request->upi_id,
            'is_default' => $isDefault,
        ]);

        return back()->with('success', 'Payment method added successfully.');
    }

    public function destroy(UserPaymentMethod $paymentMethod)
    {
        if ($paymentMethod->user_id !== Auth::id()) {
            abort(403);
        }

        $wasDefault = $paymentMethod->is_default;
        $paymentMethod->delete();

        // If we deleted the default, make the most recent one default (or none)
        if ($wasDefault) {
            $newDefault = Auth::user()->paymentMethods()->latest()->first();
            if ($newDefault) {
                $newDefault->update(['is_default' => true]);
            }
        }

        return back()->with('success', 'Payment method deleted successfully.');
    }

    public function setDefault(UserPaymentMethod $paymentMethod)
    {
        if ($paymentMethod->user_id !== Auth::id()) {
            abort(403);
        }

        Auth::user()->paymentMethods()->update(['is_default' => false]);
        $paymentMethod->update(['is_default' => true]);

        return back()->with('success', 'Default payment method updated.');
    }
}
