<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderReturn;
use App\Models\GeneralSetting;
use Illuminate\Http\Request;
use Carbon\Carbon;

class OrderReturnController extends Controller
{
    public function store(Request $request, Order $order)
    {
        $request->validate([
            'reason' => 'required|string|min:10',
        ]);

        if ($order->user_id !== auth()->id()) {
            abort(403);
        }

        if ($order->status !== 'completed') {
            return back()->with('error', 'Only completed orders can be returned.');
        }

        if ($order->returnRequest) {
            return back()->with('error', 'A return request already exists for this order.');
        }

        // Check eligibility
        foreach ($order->items as $item) {
            $product = $item->product;

            if (!$product)
                continue; // Should not happen but safety

            if (!$product->is_returnable) {
                return back()->with('error', 'This order contains non-returnable items (' . $product->name . ').');
            }

            // Check return window
            $window = $product->return_window_days;
            if (is_null($window)) {
                $window = GeneralSetting::value('return_window_days') ?? 7;
            }

            // Determine completion date (approximate using updated_at if status is completed)
            // Ideally we check status history, but updated_at is a good proxy if it hasn't changed since.
            // Better: Find when it became completed.
            $completedAt = $order->updated_at; // Fallback
            $history = $order->statusHistories()->where('status', 'completed')->latest()->first();
            if ($history) {
                $completedAt = $history->created_at;
            }

            if ($completedAt->addDays($window)->isPast()) {
                return back()->with('error', 'Return window for ' . $product->name . ' has closed.');
            }
        }

        OrderReturn::create([
            'order_id' => $order->id,
            'user_id' => auth()->id(),
            'reason' => $request->reason,
            'status' => 'requested',
        ]);

        return back()->with('success', 'Return requested successfully.');
    }
}
