<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class ShopController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('status', true);

        if ($request->has('search') && $request->search != '') {
            $query->where('name', 'like', '%' . $request->search . '%');

            if (auth()->check()) {
                \App\Models\SearchHistory::updateOrCreate(
                    ['user_id' => auth()->id(), 'query' => $request->search],
                    ['updated_at' => now()]
                );
            }
        }

        // Filter by Categories
        if ($request->has('categories')) {
            $query->whereIn('category_id', $request->input('categories'));
        }

        // Filter by Price
        if ($request->has('min_price') && $request->has('max_price')) {
            $min = $request->input('min_price');
            $max = $request->input('max_price');
            $query->whereHas('variations', function ($q) use ($min, $max) {
                $q->whereBetween('sp', [$min, $max]);
            });
        }

        // Sorting
        $sort = $request->input('sort', 'latest');

        switch ($sort) {
            case 'price_low':
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MIN(product_variations.sp) as min_price'))
                    ->groupBy('products.id')
                    ->orderBy('min_price', 'asc');
                break;

            case 'price_high':
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MAX(product_variations.sp) as max_price'))
                    ->groupBy('products.id')
                    ->orderBy('max_price', 'desc');
                break;

            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;

            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;

            case 'latest':
            default:
                $query->latest();
                break;
        }

        $products = $query->with(['images', 'variations'])->paginate(50);

        // Data for filters
        $categories = Category::whereNull('parent_id')
            ->where('status', true)
            ->withCount([
                'products' => function ($query) {
                    $query->where('status', true);
                }
            ])
            ->having('products_count', '>', 0)
            ->get();

        $priceStats = \App\Models\ProductVariation::selectRaw('min(sp) as min_price, max(sp) as max_price')->first();
        $minPrice = $priceStats->min_price ?? 0;
        $maxPrice = $priceStats->max_price ?? 10000;

        // Pass search query to view to persist in input if on search page
        $search = $request->search;

        $recentSearches = auth()->check() ? \App\Models\SearchHistory::where('user_id', auth()->id())->latest()->take(5)->pluck('query') : collect();

        $settings = \App\Models\GeneralSetting::firstOrNew();

        // If AJAX request, return only products HTML
        if ($request->ajax()) {
            return response()->json([
                'html' => view('shop.partials.product-grid', compact('products'))->render(),
                'has_more' => $products->hasMorePages(),
                'next_page' => $products->currentPage() + 1
            ]);
        }

        return view('shop.index', compact('products', 'search', 'categories', 'minPrice', 'maxPrice', 'recentSearches', 'settings'));
    }

    public function searchSuggestions(Request $request)
    {
        $query = $request->get('query');
        if (strlen($query) < 2) {
            return response()->json([]);
        }

        $products = Product::where('status', true)
            ->where('name', 'like', '%' . $query . '%')
            ->select('id', 'name', 'slug', 'sp')
            ->with([
                'images' => function ($q) {
                    $q->orderBy('sort_order')->limit(1);
                },
                'variations' => function ($q) {
                    $q->with('options')->orderBy('sort_order');
                }
            ])
            ->limit(5)
            ->get();

        $results = collect();

        foreach ($products as $product) {
            // Determine Product Image
            $productImage = null;
            if ($product->images->isNotEmpty()) {
                $productImage = $product->images->first()->image_path;
            } elseif ($product->variations->isNotEmpty()) {
                $productImage = $product->variations->first()->image_path;
            } else {
                $productImage = 'https://placehold.co/50x50?text=No+Img';
            }

            // Fix for null price: use variation min price OR fallback to product sp
            $productPrice = $product->variations->isNotEmpty() ? $product->variations->min('sp') : $product->sp;

            // Add Main Product
            $results->push([
                'name' => $product->name,
                'url' => route('shop.product', $product->slug),
                'image' => $productImage,
                'price' => $productPrice
            ]);

            // Add Variations if exists
            if ($product->variations->count() > 0) {
                foreach ($product->variations as $variation) {
                    // Construct Name from Options
                    $optionsName = $variation->options->map(function ($opt) {
                        return $opt->attribute_value;
                    })->join(', ');

                    $variationName = $product->name . ($optionsName ? ' - ' . $optionsName : '');

                    // Variation Image
                    $varImage = $variation->image_path ?? $productImage;

                    $results->push([
                        'name' => $variationName,
                        'url' => route('shop.product', $product->slug),
                        'image' => $varImage,
                        'price' => $variation->sp
                    ]);
                }
            }
        }

        return response()->json($results);
    }

    public function category(Request $request, $slug)
    {
        $category = Category::where('slug', $slug)
            ->where('status', true)
            ->firstOrFail();

        // Get products from this category AND its subcategories
        $categoryIds = $category->children()->pluck('id')->push($category->id);

        $query = Product::where(function ($q) use ($categoryIds) {
            $q->whereIn('category_id', $categoryIds)
                ->orWhereIn('subcategory_id', $categoryIds);
        })->where('status', true);

        // Filter by Subcategories
        if ($request->has('subcategories')) {
            $query->whereIn('subcategory_id', $request->input('subcategories'));
        }

        // Filter by Price
        if ($request->has('min_price') && $request->has('max_price')) {
            $min = $request->input('min_price');
            $max = $request->input('max_price');
            $query->whereHas('variations', function ($q) use ($min, $max) {
                $q->whereBetween('sp', [$min, $max]);
            });
        }

        // Sorting
        $sort = $request->input('sort', 'latest');

        switch ($sort) {
            case 'price_low':
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MIN(product_variations.sp) as min_price'))
                    ->groupBy('products.id')
                    ->orderBy('min_price', 'asc');
                break;

            case 'price_high':
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MAX(product_variations.sp) as max_price'))
                    ->groupBy('products.id')
                    ->orderBy('max_price', 'desc');
                break;

            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;

            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;

            case 'latest':
            default:
                $query->latest();
                break;
        }

        $products = $query->with(['images', 'variations'])->paginate(50);

        // Calculate Min and Max Price for the slider (based on current category context)
        // We get the IDs of all products in this category context primarily to check variations
        // Efficient way:
        $categoryProductsQuery = Product::where(function ($q) use ($categoryIds) {
            $q->whereIn('category_id', $categoryIds)
                ->orWhereIn('subcategory_id', $categoryIds);
        })->where('status', true);

        $productIds = $categoryProductsQuery->pluck('id');

        $priceStats = \App\Models\ProductVariation::whereIn('product_id', $productIds)
            ->selectRaw('min(sp) as min_price, max(sp) as max_price')
            ->first();

        $minPrice = $priceStats ? ($priceStats->min_price ?? 0) : 0;
        $maxPrice = $priceStats ? ($priceStats->max_price ?? 10000) : 10000;

        $subcategories = $category->children()
            ->withCount([
                'products' => function ($query) {
                    $query->where('status', true);
                }
            ])
            ->having('products_count', '>', 0)
            ->get();

        $recentSearches = auth()->check() ? \App\Models\SearchHistory::where('user_id', auth()->id())->latest()->take(5)->pluck('query') : collect();

        $settings = \App\Models\GeneralSetting::firstOrNew();

        // If AJAX request, return only products HTML
        if ($request->ajax()) {
            return response()->json([
                'html' => view('shop.partials.product-grid', compact('products'))->render(),
                'has_more' => $products->hasMorePages(),
                'next_page' => $products->currentPage() + 1
            ]);
        }

        return view('shop.index', compact('products', 'category', 'subcategories', 'minPrice', 'maxPrice', 'recentSearches', 'settings'));
    }

    public function subcategory(Request $request, $categorySlug, $subcategorySlug)
    {
        $category = Category::where('slug', $categorySlug)
            ->where('status', true)
            ->firstOrFail();

        $subcategory = Category::where('slug', $subcategorySlug)
            ->where('parent_id', $category->id)
            ->where('status', true)
            ->firstOrFail();

        $query = Product::where('subcategory_id', $subcategory->id)
            ->where('status', true);

        // Filter by Price
        if ($request->has('min_price') && $request->has('max_price')) {
            $min = $request->input('min_price');
            $max = $request->input('max_price');
            $query->whereHas('variations', function ($q) use ($min, $max) {
                $q->whereBetween('sp', [$min, $max]);
            });
        }

        // Sorting
        $sort = $request->input('sort', 'latest');

        switch ($sort) {
            case 'price_low':
                // Sort by lowest price (using minimum variation price)
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MIN(product_variations.sp) as min_price'))
                    ->groupBy('products.id')
                    ->orderBy('min_price', 'asc');
                break;

            case 'price_high':
                // Sort by highest price (using maximum variation price)
                $query->join('product_variations', 'products.id', '=', 'product_variations.product_id')
                    ->select('products.*', \DB::raw('MAX(product_variations.sp) as max_price'))
                    ->groupBy('products.id')
                    ->orderBy('max_price', 'desc');
                break;

            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;

            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;

            case 'latest':
            default:
                $query->latest();
                break;
        }

        $products = $query->with(['images', 'variations'])
            ->paginate(50);

        // Calculate Min and Max Price for this subcategory
        $productIds = Product::where('subcategory_id', $subcategory->id)
            ->where('status', true)
            ->pluck('id');

        $priceStats = \App\Models\ProductVariation::whereIn('product_id', $productIds)
            ->selectRaw('min(sp) as min_price, max(sp) as max_price')
            ->first();

        $minPrice = $priceStats ? ($priceStats->min_price ?? 0) : 0;
        $maxPrice = $priceStats ? ($priceStats->max_price ?? 10000) : 10000;

        $recentSearches = auth()->check() ? \App\Models\SearchHistory::where('user_id', auth()->id())->latest()->take(5)->pluck('query') : collect();

        $settings = \App\Models\GeneralSetting::firstOrNew();

        // If AJAX request, return only products HTML
        if ($request->ajax()) {
            return response()->json([
                'html' => view('shop.partials.product-grid', compact('products'))->render(),
                'has_more' => $products->hasMorePages(),
                'next_page' => $products->currentPage() + 1
            ]);
        }

        return view('shop.index', compact('products', 'category', 'subcategory', 'minPrice', 'maxPrice', 'recentSearches', 'settings'));
    }
    public function show($slug)
    {
        $product = Product::where('slug', $slug)
            ->where('status', true)
            ->with(['images', 'variations.options', 'category', 'subcategory', 'publishedReviews.user'])
            ->firstOrFail();

        $upsellProducts = $product->upsells()
            ->where('status', true)
            ->with(['images', 'variations'])
            ->take(4)
            ->get();

        // If not enough upsells, grab some from same category as fallback/addition
        if ($upsellProducts->count() < 4) {
            $related = Product::where('category_id', $product->category_id)
                ->where('id', '!=', $product->id)
                ->where('status', true)
                ->with(['images', 'variations'])
                ->take(4 - $upsellProducts->count())
                ->get();
            $upsellProducts = $upsellProducts->concat($related);
        }

        $settings = \App\Models\GeneralSetting::firstOrNew();

        return view('shop.show', compact('product', 'upsellProducts', 'settings'));
    }
}
