<?php

namespace App\Http\Controllers\Reseller;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Order;
use App\Models\OrderItem;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::where('user_id', Auth::id());

        // Status Filtering
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        $orders = $query->latest()->paginate(10);

        // Calculate Counts
        $counts = [
            'all' => Order::where('user_id', Auth::id())->count(),
            'pending' => Order::where('user_id', Auth::id())->where('status', 'pending')->count(),
            'quotation_sent' => Order::where('user_id', Auth::id())->where('status', 'quotation_sent')->count(),
            'quotation_accepted' => Order::where('user_id', Auth::id())->where('status', 'quotation_accepted')->count(),
            'completed' => Order::where('user_id', Auth::id())->where('status', 'completed')->count(),
        ];

        return view('reseller.orders.index', compact('orders', 'counts'));
    }

    public function create(Request $request)
    {
        $query = Product::with(['category', 'subcategory', 'images', 'variations'])->where('status', true);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%");
            });
        }

        if ($request->has('category')) {
            $query->where('category_id', $request->category);
        }

        $perPage = $request->input('per_page', 20);
        if ($perPage === 'all') {
            $perPage = $query->count();
            $perPage = $perPage > 0 ? $perPage : 1; // Avoid division by zero if empty
        }

        $products = $query->paginate($perPage)->appends($request->query());
        $categories = Category::whereNull('parent_id')->get();

        return view('reseller.orders.create', compact('products', 'categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        $totalAmount = 0;
        $orderItemsData = [];

        foreach ($request->items as $item) {
            if (isset($item['quantity']) && $item['quantity'] > 0) {
                $product = Product::find($item['product_id']);
                // Use Reseller Price if available, else SP, else MRP
                $price = $product->reseller_price ?: ($product->sp ?: $product->mrp);

                $totalAmount += $price * $item['quantity'];

                $orderItemsData[] = [
                    'product_id' => $product->id,
                    'price' => $price,
                    'quantity' => $item['quantity'],
                    'total' => $price * $item['quantity'],
                ];
            }
        }

        if (empty($orderItemsData)) {
            return back()->with('error', 'Please select at least one product.');
        }

        $order = DB::transaction(function () use ($totalAmount, $orderItemsData) {
            $reseller = \App\Models\Reseller::where('user_id', Auth::id())->first();

            $address = \App\Models\Address::create([
                'user_id' => Auth::id(),
                'name' => Auth::user()->name,
                'phone' => $reseller->phone,
                'address_line1' => $reseller->address,
                'city' => $reseller->city,
                'state' => $reseller->state,
                'country' => $reseller->country,
                'postal_code' => $reseller->pincode,
                'type' => 'Home', // default
                'is_default' => false
            ]);

            $order = Order::create([
                'user_id' => Auth::id(),
                'order_number' => 'ORD-' . strtoupper(uniqid()),
                'status' => 'pending',
                'payment_status' => 'pending',
                'payment_method' => 'cod',
                'subtotal' => $totalAmount,
                'total' => $totalAmount,
                'shipping_address_id' => $address->id,
                'billing_address_id' => $address->id,
            ]);

            foreach ($orderItemsData as $itemData) {
                $order->items()->create($itemData);
            }

            return $order;
        });

        return redirect()->route('reseller.orders.index')->with('success', 'Order placed successfully.')->with('new_order_id', $order->id);
    }

    public function show(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }
        $order->load(['items.product', 'shippingAddress']);
        return view('reseller.orders.show', compact('order'));
    }

    public function acceptQuotation(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        if ($order->status !== 'quotation_sent') {
            return back()->with('error', 'This order is not pending quotation acceptance.');
        }

        $order->update(['status' => 'quotation_accepted']);

        // Optional: Send Notification to Admin

        return back()->with('success', 'Quotation accepted successfully. The order is now processing.');
    }

    public function downloadInvoice(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $order->load(['items.product', 'items.variation', 'shippingAddress', 'billingAddress', 'user']);
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.invoice', compact('order'));

        $filename = ($order->status === 'quotation_sent' ? 'quotation-' : 'invoice-') . $order->order_number . '.pdf';

        return $pdf->stream($filename);
    }
}
