<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index()
    {
        $categories = Category::whereNull('parent_id')
            ->where('status', true)
            ->where(function ($query) {
                // Has direct products
                $query->whereHas('products', function ($q) {
                    $q->where('status', true);
                })
                    // OR has child categories with products
                    ->orWhereHas('children', function ($q) {
                    $q->where('status', true)
                        ->whereHas('subProducts', function ($sq) {
                            $sq->where('status', true);
                        });
                });
            })
            ->with([
                'children' => function ($query) {
                    $query->where('status', true)
                        ->whereHas('subProducts', function ($sq) {
                            $sq->where('status', true);
                        })
                        ->with([
                            'subProducts' => function ($q) {
                                $q->where('status', true)->take(4)->with('images');
                            }
                        ]);
                },
                'products' => function ($q) {
                    $q->where('status', true)->take(4)->with('images');
                }
            ])
            ->get();

        // Best Sellers Logic
        // Fetch top 4 products sold in terms of quantity in the last 30 days
        // We look at OrderItems for orders that are not 'pending_payment'
        // Group by product_id and sum quantity
        $bestSellers = Product::join(
            DB::raw('
            (
                SELECT 
                    order_items.product_id,
                    SUM(order_items.quantity) AS total_sold
                FROM order_items
                INNER JOIN orders 
                    ON orders.id = order_items.order_id
                WHERE orders.created_at >= "' . now()->subDays(30) . '"
                  AND orders.status != "pending_payment"
                GROUP BY order_items.product_id
                ORDER BY total_sold DESC
                LIMIT 4
            ) AS best_selling
        '),
            'products.id',
            '=',
            'best_selling.product_id'
        )
            ->where('products.status', true)
            ->orderByDesc('best_selling.total_sold')
            ->with('images')
            ->select('products.*', 'best_selling.total_sold')
            ->get();

        // Fallback: If no sales data (new store), just pick random 4 featured/active products
        if ($bestSellers->isEmpty()) {
            $bestSellers = Product::where('status', true)->inRandomOrder()->take(4)->with('images')->get();
        }

        $sliders = \App\Models\Slider::where('status', true)->orderBy('sort_order')->get();

        $recentBlogs = \App\Models\BlogPost::where('status', 'published')
            ->latest('published_at')
            ->take(3)
            ->get();

        $testimonials = \App\Models\Testimonial::where('is_active', true)->latest()->take(10)->get();
        
        // Fetch subcategories for video consultation form
        $subcategories = Category::whereNotNull('parent_id')->where('status', true)->orderBy('name')->get();

        return view('home', compact('categories', 'sliders', 'recentBlogs', 'testimonials', 'bestSellers', 'subcategories'));
    }
}
