<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\DeliveryAgent;
use Illuminate\Support\Facades\Hash;

class DeliveryAgentController extends Controller
{
    public function dashboard()
    {
        $user = Auth::user();
        if (!$user->hasRole('delivery_agent')) {
            return redirect()->route('home')->with('error', 'Access denied.');
        }

        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        // Stats
        $newOrders = Order::where('delivery_agent_id', $agent->id)->whereIn('status', ['pending', 'processing'])->count();
        $inTransit = Order::where('delivery_agent_id', $agent->id)->where('status', 'shipped')->count();
        $completed = Order::where('delivery_agent_id', $agent->id)->where('status', 'completed')->count();
        $cancelled = Order::where('delivery_agent_id', $agent->id)->where('status', 'cancelled')->count();

        $cashCollected = Order::where('delivery_agent_id', $agent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->sum('total');

        $codOrderCount = Order::where('delivery_agent_id', $agent->id)
            ->where('status', 'completed')
            ->where('payment_method', 'cod')
            ->count();

        $unreadNotifications = $user->unreadNotifications->count();

        return view('delivery.dashboard', compact('agent', 'newOrders', 'inTransit', 'completed', 'cancelled', 'cashCollected', 'codOrderCount', 'unreadNotifications'));
    }

    public function profile()
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->with(['documents', 'vehicles'])->firstOrFail();
        return view('delivery.profile', compact('agent', 'user'));
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'profile_photo' => 'nullable|image|max:2048',
            'bank_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'ifsc_code' => 'nullable|string|max:255',
        ]);

        if ($request->hasFile('profile_photo')) {
            if ($user->profile_photo_path) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($user->profile_photo_path);
            }
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $user->profile_photo_path = $path;
        }

        $user->update(['name' => $request->name]);
        if ($user->isDirty('profile_photo_path')) {
            $user->save();
        }

        $agent->update([
            'phone' => $request->phone,
            'bank_name' => $request->bank_name,
            'account_number' => $request->account_number,
            'ifsc_code' => $request->ifsc_code,
        ]);

        return back()->with('success', 'Profile updated successfully.');
    }

    public function orders(Request $request)
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $status = $request->get('status', 'assigned'); // Default to assigned/new

        $query = Order::where('delivery_agent_id', $agent->id);

        switch ($status) {
            case 'assigned':
                $query->whereIn('status', ['pending', 'processing']);
                break;
            case 'transit':
                $query->where('status', 'shipped');
                break;
            case 'completed':
                $query->where('status', 'completed');
                break;
            case 'cancelled':
                $query->where('status', 'cancelled');
                break;
        }

        $assignedCount = Order::where('delivery_agent_id', $agent->id)->whereIn('status', ['pending', 'processing'])->count();
        $transitCount = Order::where('delivery_agent_id', $agent->id)->where('status', 'shipped')->count();
        $completedCount = Order::where('delivery_agent_id', $agent->id)->where('status', 'completed')->count();
        $cancelledCount = Order::where('delivery_agent_id', $agent->id)->where('status', 'cancelled')->count();

        $orders = $query->with('pickupStore')->latest()->paginate(10);

        return view('delivery.orders', compact('orders', 'status', 'agent', 'assignedCount', 'transitCount', 'completedCount', 'cancelledCount'));
    }

    public function showOrder($id)
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $order = Order::with(['pickupStore', 'agentRating', 'productReviews.product'])->where('delivery_agent_id', $agent->id)->where('id', $id)->firstOrFail();

        return view('delivery.orders.show', compact('order'));
    }

    public function updateOrderStatus(Request $request, $id)
    {
        // To accept orders (move to transit), complete, etc.
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $order = Order::where('delivery_agent_id', $agent->id)->where('id', $id)->firstOrFail();

        $request->validate([
            'status' => 'required|in:shipped,completed,cancelled',
            'reason' => 'required_if:status,cancelled|nullable|string|max:1000',
        ]);

        $previousStatus = $order->status;
        $order->update([
            'status' => $request->status,
        ]);

        if ($request->status == 'cancelled' && $request->filled('reason')) {
            $order->update(['cancellation_reason' => $request->reason]);
        }

        // Handle COD Payment & Wallet Logic (Mirroring Admin Logic)
        $orderUser = $order->user;
        if ($orderUser) {
            $settings = \App\Models\GeneralSetting::firstOrNew();

            if ($request->status == 'completed') {
                // COD Payment Logic
                if ($order->payment_method == 'cod' && $order->payment_status == 'pending') {
                    $order->update(['payment_status' => 'paid']);
                }

                // Wallet Earn Logic
                if ($settings->wallet_enabled && $order->payment_status != 'refunded') {
                    $alreadyEarned = $orderUser->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('type', 'credit')
                        ->where('description', 'like', 'Earned%')
                        ->exists();

                    if (!$alreadyEarned && $settings->wallet_earn_rate > 0) {
                        $earnPoints = floor(($order->total / 100) * $settings->wallet_earn_rate);
                        if ($earnPoints > 0) {
                            $orderUser->creditWallet($earnPoints, "Earned from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                        }
                    }
                }
            }

            if ($request->status == 'cancelled') {
                // Refer to Admin/OrderController for full logic ideally, simplified here for robustness
                // Refund Redeemed Points
                if ($settings->wallet_enabled && $order->points_redeemed > 0) {
                    $alreadyRefunded = $orderUser->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('description', 'like', 'Refund%')
                        ->exists();

                    if (!$alreadyRefunded) {
                        $orderUser->creditWallet($order->points_redeemed, "Refund for cancelled Order #" . $order->order_number, 'App\Models\Order', $order->id);
                    }
                }

                // Refund Gift Card
                if ($order->gift_card_discount > 0) {
                    // We can't credit wallet for GC easily without more logic, simpler to just mark as refunded in description if we had a dedicated transactional method or credit wallet as cash equivalent?
                    // Admin controller credits wallet for GC refund. copying that.
                    $alreadyGcRefunded = $orderUser->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('description', 'like', 'Gift Card Refund%')
                        ->exists();

                    if (!$alreadyGcRefunded) {
                        $orderUser->creditWallet($order->gift_card_discount, "Gift Card Refund for cancelled Order #" . $order->order_number, 'App\Models\Order', $order->id);
                    }
                }

                // Reverse Earned Points
                if ($settings->wallet_enabled) {
                    $earnedTransaction = $orderUser->walletTransactions()
                        ->where('reference_type', 'App\Models\Order')
                        ->where('reference_id', $order->id)
                        ->where('type', 'credit')
                        ->where('description', 'like', 'Earned%')
                        ->first();

                    if ($earnedTransaction) {
                        $alreadyReversed = $orderUser->walletTransactions()
                            ->where('reference_type', 'App\Models\Order')
                            ->where('reference_id', $order->id)
                            ->where('type', 'debit')
                            ->where('description', 'like', 'Reversal%')
                            ->exists();

                        if (!$alreadyReversed) {
                            $amountToReverse = ($orderUser->wallet_balance >= $earnedTransaction->amount) ? $earnedTransaction->amount : $orderUser->wallet_balance;
                            if ($amountToReverse > 0) {
                                $orderUser->debitWallet($amountToReverse, "Reversal of points from Order #" . $order->order_number, 'App\Models\Order', $order->id);
                            }
                        }
                    }
                }
            }
        }

        return back()->with('success', 'Order status updated.');
    }

    public function notifications()
    {
        $user = Auth::user();
        $notifications = $user->notifications()->latest()->paginate(10);
        $user->unreadNotifications->markAsRead(); // Mark all as read when viewed

        return view('delivery.notifications', compact('notifications'));
    }

    public function settings()
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        return view('delivery.settings', compact('agent'));
    }

    public function updateSettings(Request $request)
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        if ($request->has('toggle_availability')) {
            $agent->is_available = !$agent->is_available;
            $agent->save();
            return back()->with('success', 'Availability updated to ' . ($agent->is_available ? 'Available' : 'Unavailable'));
        }

        if ($request->has('update_password')) {
            $request->validate([
                'current_password' => 'required',
                'password' => 'required|min:8|confirmed'
            ]);

            if (!Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => 'Incorrect current password']);
            }

            $user->update(['password' => Hash::make($request->password)]);
            return back()->with('success', 'Password updated successfully');
        }

        if ($request->has('deactivate_account')) {
            // Logic to deactivate
            $user->update(['is_active' => false]);

            // Sync with DeliveryAgent profile status
            $agent->update(['status' => false, 'is_available' => false]);

            Auth::logout();
            return redirect()->route('login')->with('success', 'Account deactivated.');
        }
        return back();
    }

    public function updateLocation(Request $request)
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();

        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        $agent->update([
            'current_latitude' => $request->latitude,
            'current_longitude' => $request->longitude,
        ]);

        return response()->json(['success' => true]);
    }
    public function wallet()
    {
        $user = Auth::user();
        $agent = DeliveryAgent::where('user_id', $user->id)->firstOrFail();
        $transactions = $user->walletTransactions()->paginate(10);
        $settings = \App\Models\GeneralSetting::firstOrNew();

        return view('delivery.wallet', compact('agent', 'transactions', 'settings', 'user'));
    }
}
