<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ChatbotSetting;
use App\Models\ChatbotKnowledgeBase;
use App\Models\Product;
use App\Models\Store;
use Illuminate\Support\Str;

class ChatbotController extends Controller
{
    public function index()
    {
        $settings = ChatbotSetting::firstOrCreate(['id' => 1], [
            'name' => 'Support Bot',
            'color_scheme' => '#9B2D69',
            'welcome_message' => 'Hello! How can I help you today?',
            'is_active' => true,
        ]);

        return response()->json([
            'settings' => $settings
        ]);
    }

    public function sendMessage(Request $request)
    {
        $request->validate([
            'message' => 'required|string',
        ]);

        $message = strtolower(trim($request->message));
        $generalSettings = \App\Models\GeneralSetting::first();

        // --- 1. Small Talk / Common Greetings ---
        $greetings = [
            'hi' => "Hello! How can I help you with your hair needs today?",
            'hello' => "Hi there! Welcome to Hair Niche.",
            'hey' => "Hey! Looking for a specific hair system or product?",
            'how are you' => "I'm just a bot, but I'm ready to help you find the perfect look! How can I assist?",
            'who are you' => "I am your personal shopping assistant for Hair Niche.",
            'age' => "I'm brand new and always learning!",
            'thank you' => "You're welcome! Happy shopping.",
            'thanks' => "No problem! Let me know if you need anything else.",
            'bye' => "Goodbye! Have a great hair day!",
            'good morning' => "Good morning! How can I start your day?",
            'good evening' => "Good evening! How can I help?",
        ];

        foreach ($greetings as $key => $reply) {
            if ($message === $key || preg_match('/\b' . preg_quote($key, '/') . '\b/', $message)) {
                return response()->json(['reply' => $reply, 'status' => 'success']);
            }
        }

        // --- 2. Keyword Matching (Like Operator Logic) ---
        // Fetch all KB entries that have keywords defined
        $kbEntries = ChatbotKnowledgeBase::whereNotNull('keywords')->get();
        foreach ($kbEntries as $entry) {
            $keywords = explode(',', $entry->keywords);
            foreach ($keywords as $keyword) {
                $keyword = strtolower(trim($keyword));
                if (!empty($keyword) && (str_contains($message, $keyword) || preg_match('/\b' . preg_quote($keyword, '/') . '\b/', $message))) {
                    return response()->json(['reply' => $entry->answer, 'status' => 'success']);
                }
            }
        }

        // --- 3. Dynamic "Call" Handle ---
        if (str_contains($message, 'call') || str_contains($message, 'phone') || str_contains($message, 'contact') || str_contains($message, 'mobile')) {
            $phone = $generalSettings->phone_sales ?? '+91-XXXXX-XXXXX';
            return response()->json([
                'reply' => "You can call us directly at <strong>{$phone}</strong> for immediate assistance. We are available between 10 AM to 7 PM.",
                'status' => 'success'
            ]);
        }

        // --- 4. Knowledge Base Search (Exact/Topic Matches) ---
        $knowledge = ChatbotKnowledgeBase::where('question', 'LIKE', "%{$message}%")
            ->orWhere('topic', 'LIKE', "%{$message}%")
            ->first();

        if ($knowledge) {
            return response()->json(['reply' => $knowledge->answer, 'status' => 'success']);
        }

        // --- 5. Dynamic Product Knowledge & Similar Keywords ---
        if (strlen($message) > 2) {
            $products = Product::where('name', 'LIKE', "%{$message}%")
                ->orWhere('description', 'LIKE', "%{$message}%")
                ->where('status', true)
                ->take(3)
                ->get();

            if ($products->count() > 0) {
                $reply = "I found these products that might match '{$message}':<br>";
                foreach ($products as $product) {
                    $url = route('shop.product', $product->slug);
                    $reply .= "- <a href='{$url}' class='text-blue-300 underline font-semibold hover:text-white' target='_blank'>{$product->name}</a><br>";
                }
                $reply .= "<br>Click on any link to view details.";
                return response()->json(['reply' => $reply, 'status' => 'success']);
            }
        }

        // --- 6. Locations Knowledge ---
        if (str_contains($message, 'location') || str_contains($message, 'store') || str_contains($message, 'address') || str_contains($message, 'visit')) {
            $stores = Store::where('is_active', true)->get();
            if ($stores->count() > 0) {
                $reply = "You can visit us at the following locations:<br><br>";
                foreach ($stores as $store) {
                    $addr = $store->address_line1 . ($store->address_line2 ? ', ' . $store->address_line2 : '');
                    $reply .= "<b>{$store->name}</b><br>{$addr}, {$store->city}<br>Phone: {$store->phone}<br><br>";
                }
                return response()->json(['reply' => $reply, 'status' => 'success']);
            } else {
                return response()->json(['reply' => "We are an online store, but we might have partner locations soon. Please check our Contact page for details.", 'status' => 'success']);
            }
        }

        // --- 7. Video Consultation Knowledge ---
        if (str_contains($message, 'video') || str_contains($message, 'consultation') || str_contains($message, 'book')) {
            return response()->json([
                'reply' => "<strong>How to Book a Video Consultation:</strong><br><br>1. We offer personalized video consultations for hair systems and patches.<br>2. To book, please call or WhatsApp our support line at " . ($generalSettings->phone_sales ?? '+91-XXXXX-XXXXX') . ".<br>3. Or send us a message via the 'Contact Us' page requesting a slot.<br>4. Our experts will guide you through measurements and selection.",
                'status' => 'success'
            ]);
        }

        // --- 6. Extended Keyword Search in KB (Fallback) ---
        $keywords = explode(' ', $message);
        foreach ($keywords as $keyword) {
            if (strlen($keyword) > 3) {
                $knowledge = ChatbotKnowledgeBase::where('question', 'LIKE', "%{$keyword}%")
                    ->orWhere('topic', 'LIKE', "%{$keyword}%")
                    ->orWhere('answer', 'LIKE', "%{$keyword}%")
                    ->first();
                if ($knowledge) {
                    return response()->json(['reply' => $knowledge->answer, 'status' => 'success']);
                }
            }
        }

        // --- 7. Final Fallback ---
        return response()->json([
            'reply' => "I'm not sure about that. Try asking about 'hair systems', 'extensions', 'order status', or specific products. You can also contact our support team directly.",
            'status' => 'success'
        ]);
    }
}
