<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\CartItem;
use App\Models\Product;
use App\Models\ProductVariation;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Services\ActivityTracker;

class CartController extends Controller
{

    public function index()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Cart functionality is disabled for delivery agents.');
        }
        $cart = $this->getCart();

        // -------------------------------------------------------------
        // INVENTORY CHECK LOGIC (Added based on User Request)
        // -------------------------------------------------------------
        if ($cart && $cart->items->count() > 0) {
            $messages = [];
            $hasChanges = false;
            foreach ($cart->items as $item) {
                // Remove if product deleted or inactive
                if (!$item->product || $item->product->status == 0) {
                    $messages[] = "Item '" . ($item->product ? $item->product->name : 'Unknown') . "' was removed because it is no longer available.";
                    $item->delete();
                    $hasChanges = true;
                    continue;
                }

                // Variation check
                if ($item->variation) {
                    // Update Price if changed
                    if ($item->price != $item->variation->sp) {
                        $item->price = $item->variation->sp;
                        $item->save();
                    }

                    // Stock Check
                    if ($item->variation->stock <= 0) {
                        $messages[] = "Item '" . $item->product->name . "' was removed because the selected variation is out of stock.";
                        $item->delete();
                        $hasChanges = true;
                    } elseif ($item->variation->stock < $item->quantity) {
                        $messages[] = "Quantity for '" . $item->product->name . "' was reduced to " . $item->variation->stock . " due to available stock.";
                        $item->quantity = $item->variation->stock;
                        $item->save();
                        $hasChanges = true;
                    }
                } else {
                    // Update Price if changed
                    if ($item->price != $item->product->sp) {
                        $item->price = $item->product->sp;
                        $item->save();
                    }
                    // No stock check column on products table found earlier, but if it exists or if we ignore it:
                    // Assuming 'products' has no 'stock' based on earlier checks.
                    // If needed later, add general product stock check here.
                }
            }

            if ($hasChanges) {
                // Reload cart items to reflect deletions/updates
                $cart->load('items');
            }

            if (count($messages) > 0) {
                session()->flash('warning', implode(' ', $messages));
            }
        }
        // -------------------------------------------------------------


        // Logic to fetch cross-sell products
        $crossSellProducts = collect();

        if ($cart && $cart->items->count() > 0) {
            $cartProductIds = $cart->items->pluck('product_id')->unique();
            $cartCategoryIds = [];
            $cartSubcategoryIds = [];

            // 1. Explicit Cross-Sells
            // We iterate to respect individual product relationships or use a whereIn query on pivot if structure allowed, 
            // but standard relation access is per model. optimize with eager load if needed, but for cart size < 10 loop is fine.
            foreach ($cart->items as $item) {
                if ($item->product) {
                    $cartCategoryIds[] = $item->product->category_id;
                    if ($item->product->subcategory_id) {
                        $cartSubcategoryIds[] = $item->product->subcategory_id;
                    }

                    $related = $item->product->crossSells()->limit(4)->get();
                    $crossSellProducts = $crossSellProducts->merge($related);
                }
            }

            // Remove items already in cart
            $crossSellProducts = $crossSellProducts->whereNotIn('id', $cartProductIds); // Collection filtering

            // 2. Fallback: Same Subcategory
            if ($crossSellProducts->count() < 4 && !empty($cartSubcategoryIds)) {
                $needed = 4 - $crossSellProducts->count();
                $moreProducts = Product::whereIn('subcategory_id', $cartSubcategoryIds)
                    ->whereNotIn('id', $cartProductIds)
                    ->whereNotIn('id', $crossSellProducts->pluck('id'))
                    ->where('status', true)
                    ->inRandomOrder()
                    ->take($needed)
                    ->get();
                $crossSellProducts = $crossSellProducts->merge($moreProducts);
            }

            // 3. Fallback: Same Category
            if ($crossSellProducts->count() < 4 && !empty($cartCategoryIds)) {
                $needed = 4 - $crossSellProducts->count();
                $moreProducts = Product::whereIn('category_id', $cartCategoryIds)
                    ->whereNotIn('id', $cartProductIds)
                    ->whereNotIn('id', $crossSellProducts->pluck('id'))
                    ->where('status', true)
                    ->inRandomOrder()
                    ->take($needed)
                    ->get();
                $crossSellProducts = $crossSellProducts->merge($moreProducts);
            }
        }

        // Unique items only 
        $crossSellProducts = $crossSellProducts->unique('id')->take(4);

        return view('shop.cart', compact('cart', 'crossSellProducts'));
    }

    public function add(Request $request)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Cart functionality is disabled for delivery agents.');
        }

        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'variation_id' => 'nullable|exists:product_variations,id',
        ]);

        $cart = $this->getCart();

        $productId = $request->product_id;
        $variationId = $request->variation_id;
        $quantity = $request->quantity;

        $product = Product::findOrFail($productId);
        $price = $product->sp; // Default product SP (if no variation or as fallback)

        if ($variationId) {
            $variation = ProductVariation::findOrFail($variationId);
            $price = $variation->sp;

            // Check stock
            if ($variation->stock < $quantity) {
                return back()->with('error', 'Requested quantity exceeds available stock.');
            }
        }

        // Check for existing item
        $existingItem = $cart->items()
            ->where('product_id', $productId)
            ->where('product_variation_id', $variationId)
            ->first();

        if ($existingItem) {
            $existingItem->quantity += $quantity;
            $existingItem->save();
        } else {
            $cart->items()->create([
                'product_id' => $productId,
                'product_variation_id' => $variationId,
                'quantity' => $quantity,
                'price' => $price,
            ]);
        }

        // Log Activity
        ActivityTracker::log('add_to_cart', ['product_id' => $productId, 'quantity' => $quantity]);

        // If it was an AJAX request, we could return JSON, for now redirect
        return redirect()->route('cart.index')->with('success', 'Product added to cart!');
    }

    public function update(Request $request, $itemId)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Cart functionality is disabled for delivery agents.');
        }

        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        $cart = $this->getCart();
        $item = $cart->items()->findOrFail($itemId);

        // Stock check
        if ($item->variation) {
            if ($item->variation->stock < $request->quantity) {
                return back()->with('error', 'Requested quantity exceeds available stock.');
            }
        }

        $item->quantity = $request->quantity;
        $item->save();

        return back()->with('success', 'Cart updated.');
    }

    public function remove($itemId)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Cart functionality is disabled for delivery agents.');
        }

        $cart = $this->getCart();
        $cart->items()->where('id', $itemId)->delete();

        return back()->with('success', 'Item removed from cart.');
    }

    public function checkoutCheck()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Checkout is disabled for delivery agents.');
        }

        if (!Auth::check()) {
            session()->put('url.intended', route('cart.index'));
            return redirect()->route('login');
        }

        // If logged in, proceed to actual checkout
        return redirect()->route('checkout.index');
    }

    public function checkout()
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Checkout is disabled for delivery agents.');
        }

        // Placeholder for checkout page
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();
        $addresses = $user->addresses()->orderBy('is_default', 'desc')->get();
        $settings = \App\Models\GeneralSetting::firstOrNew();

        $coupons = \App\Models\Coupon::where('is_active', true)
            ->where(function ($query) {
                $query->whereNull('valid_from')->orWhere('valid_from', '<=', now());
            })
            ->where(function ($query) {
                $query->whereNull('valid_until')->orWhere('valid_until', '>=', now());
            })
            // Check usage limit if set (> 0)
            ->where(function ($query) {
                $query->where('usage_limit', 0)
                    ->orWhereRaw('used_count < usage_limit');
            })
            ->get();

        // Wallet Logic
        $walletBalance = $user->wallet_balance ?? 0;
        $maxRedeemablePoints = 0;
        $maxRedeemValue = 0;
        if ($settings->wallet_enabled && $walletBalance >= ($settings->wallet_min_points_redeem ?? 0)) {
            $maxRedeemablePoints = $walletBalance;
            if (($settings->wallet_exchange_rate ?? 0) > 0) {
                // Calculate value in currency
                $maxRedeemValue = $maxRedeemablePoints / $settings->wallet_exchange_rate;
            }
        }

        $currentStore = null;
        if ($settings->current_store_id) {
            $currentStore = \App\Models\Store::find($settings->current_store_id);
        }

        return view('shop.checkout', [
            'cart' => $this->getCart(),
            'user' => $user,
            'addresses' => $addresses,
            'settings' => $settings,
            'coupons' => $coupons,
            'walletBalance' => $walletBalance,
            'maxRedeemablePoints' => $maxRedeemablePoints,
            'maxRedeemValue' => $maxRedeemValue,
            'currentStore' => $currentStore
        ]);
    }

    public function validateCoupon(Request $request)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return response()->json(['valid' => false, 'message' => 'Not allowed for delivery agents.']);
        }

        $request->validate([
            'coupon_code' => 'required|string',
            'subtotal' => 'required|numeric'
        ]);

        $code = strtoupper(trim($request->coupon_code));
        $subtotal = $request->subtotal;

        $coupon = \App\Models\Coupon::where('code', $code)->first();

        if (!$coupon) {
            return response()->json([
                'valid' => false,
                'message' => 'Invalid coupon code.'
            ]);
        }

        if (!$coupon->is_active) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon is no longer active.'
            ]);
        }

        $now = now();
        if ($coupon->valid_from && $now->lt($coupon->valid_from)) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon is not yet valid.'
            ]);
        }

        if ($coupon->valid_until && $now->gt($coupon->valid_until)) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon has expired.'
            ]);
        }

        if ($coupon->usage_limit > 0 && $coupon->used_count >= $coupon->usage_limit) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon usage limit has been reached.'
            ]);
        }

        if ($coupon->min_spend > 0 && $subtotal < $coupon->min_spend) {
            return response()->json([
                'valid' => false,
                'message' => 'Minimum spend of ₹' . number_format($coupon->min_spend, 2) . ' required.'
            ]);
        }

        // Calculate discount
        $discountAmount = 0;

        if ($coupon->type === 'percentage') {
            $discountAmount = ($subtotal * $coupon->value) / 100;
        } elseif ($coupon->type === 'fixed') {
            $discountAmount = $coupon->value;
        } elseif ($coupon->type === 'buy_x_get_y') {
            // Logic: Buy X (min_spend as quantity), Get Y (value as quantity) Free
            // We need to count total items in cart or specific items? 
            // Assuming global for now as we lack product restriction columns.
            // Discount = Price of Y cheapest items for every set of (X+Y) items? 
            // Or simple: specific logic. Let's assume standard: "Buy X items, get Y cheapest items free".

            $buyQty = (int) $coupon->min_spend; // Using min_spend as 'Buy X' quantity
            $getQty = (int) $coupon->value;     // Using value as 'Get Y' quantity

            if ($buyQty > 0 && $getQty > 0) {
                // Get all cart items prices individually sorted asc
                $cart = $this->getCart();
                $allPrices = [];
                foreach ($cart->items as $item) {
                    for ($i = 0; $i < $item->quantity; $i++) {
                        $allPrices[] = $item->price;
                    }
                }
                sort($allPrices);

                $totalItems = count($allPrices);
                // How many sets involved?
                // Rule: For every (Buy + Get) items, Get items are free.
                // e.g. Buy 2 Get 1. Total 3 items. 1 cheapest is free.
                // Total 6 items. 2 cheapest are free.
                // Total 4 items. (1 set of 3). 1 cheapest free.

                $groupSize = $buyQty + $getQty;
                $numberOfFreeItems = floor($totalItems / $groupSize) * $getQty;

                // Sum up the prices of the cheapest 'numberOfFreeItems'
                for ($i = 0; $i < $numberOfFreeItems; $i++) {
                    $discountAmount += $allPrices[$i];
                }
            }
        }

        // Ensure discount doesn't exceed subtotal
        if ($discountAmount > $subtotal) {
            $discountAmount = $subtotal;
        }

        return response()->json([
            'valid' => true,
            'message' => 'Coupon applied successfully!',
            'code' => $coupon->code,
            'discount_amount' => $discountAmount,
            'type' => $coupon->type,
            'value' => $coupon->value
        ]);
    }
    public function moveToWishlist($itemId)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return redirect()->route('delivery.dashboard')->with('error', 'Wishlist functionality is disabled for delivery agents.');
        }

        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $cart = $this->getCart();
        $item = $cart->items()->where('id', $itemId)->first();

        if (!$item) {
            return back()->with('error', 'Item not found in cart.');
        }

        // Add to Wishlist if not exists
        \App\Models\Wishlist::firstOrCreate([
            'user_id' => Auth::id(),
            'product_id' => $item->product_id
        ]);

        // Remove from Cart
        $item->delete();

        return back()->with('success', 'Item moved to wishlist.');
    }

    private function getCart()
    {
        if (Auth::check()) {
            $cart = Cart::firstOrCreate(['user_id' => Auth::id()]);
            // Merge session cart if exists
            if (session()->has('cart_session_id')) {
                $sessionCart = Cart::where('session_id', session('cart_session_id'))->with('items')->first();
                if ($sessionCart) {
                    foreach ($sessionCart->items as $item) {
                        // Basic merge strategy: add to user cart
                        // Check if same product/variation exists in user cart
                        $existing = $cart->items()
                            ->where('product_id', $item->product_id)
                            ->where('product_variation_id', $item->product_variation_id)
                            ->first();

                        if ($existing) {
                            $existing->quantity += $item->quantity;
                            $existing->save();
                        } else {
                            $cart->items()->create([
                                'product_id' => $item->product_id,
                                'product_variation_id' => $item->product_variation_id,
                                'quantity' => $item->quantity,
                                'price' => $item->price
                            ]);
                        }
                    }
                    $sessionCart->delete();
                    session()->forget('cart_session_id');
                }
            }
        } else {
            $sessionId = session()->get('cart_session_id');
            if (!$sessionId) {
                $sessionId = Str::uuid()->toString();
                session()->put('cart_session_id', $sessionId);
            }
            $cart = Cart::firstOrCreate(['session_id' => $sessionId]);
        }

        return $cart;
    }
    public function validateGiftCard(Request $request)
    {
        if (Auth::check() && Auth::user()->hasRole('delivery_agent')) {
            return response()->json(['valid' => false, 'message' => 'Not allowed for delivery agents.']);
        }

        $request->validate([
            'gift_card_code' => 'required|string',
            'subtotal' => 'required|numeric'
        ]);

        $code = strtoupper(trim($request->gift_card_code));
        $subtotal = $request->subtotal;

        $giftCard = \App\Models\GiftCard::where('code', $code)->first();

        // 1. Check existence
        if (!$giftCard) {
            return response()->json(['valid' => false, 'message' => 'Invalid gift card code.']);
        }

        // 2. Check Expiry
        if ($giftCard->expires_at && now()->gt($giftCard->expires_at)) {
            return response()->json(['valid' => false, 'message' => 'This gift card has expired.']);
        }

        // 3. Check Status
        if ($giftCard->status !== 'active') {
            return response()->json(['valid' => false, 'message' => 'This gift card has already been used or is inactive.']);
        }

        // 4. Exclusivity check
        if ($request->has_coupon && $request->has_coupon == 'true') {
            return response()->json(['valid' => false, 'message' => 'Gift cards cannot be used with coupons.']);
        }

        // Calculate Discount
        // Discount applied to order is MIN(OrderTotal, GiftCardAmount).
        // If GiftCardAmount > OrderTotal, remainder will be handled in checkout process (credited to wallet).

        $discountAmount = min($subtotal, $giftCard->amount);
        $remainder = max(0, $giftCard->amount - $subtotal);

        return response()->json([
            'valid' => true,
            'message' => 'Gift card applied successfully!',
            'code' => $giftCard->code,
            'amount' => $giftCard->amount, // Full value
            'discount_amount' => $discountAmount, // Applied to cart
            'remainder' => $remainder // To be credited to wallet
        ]);
    }
}
