<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    // Show Login Form
    public function showLogin()
    {
        if (Auth::check()) {
            if (Auth::user()->hasRole('delivery_agent')) {
                return redirect()->route('delivery.dashboard');
            }
            return redirect()->route('user.dashboard');
        }
        return view('auth.login');
    }

    // Handle Login
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials)) {
            $user = Auth::user();

            if (!$user->is_active) {
                Auth::logout();
                return back()->withInput($request->only('email'))->withErrors(['email' => 'Your account is inactive.']);
            }

            $request->session()->regenerate();
            Log::info("User logged in: " . $user->email);

            if ($user->hasRole('delivery_agent')) {
                return redirect()->route('delivery.dashboard');
            }

            return redirect()->intended(route('user.dashboard'));
        }

        return back()->withInput($request->only('email'))->withErrors(['email' => 'Invalid credentials.']);
    }

    // Show Register Form
    public function showRegister()
    {
        if (Auth::check()) {
            return redirect()->route('user.dashboard');
        }
        return view('auth.register');
    }

    // Handle Register
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'mobile' => 'required|string|max:20',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'mobile' => $request->mobile,
            'whatsapp_number' => $request->mobile, // Default to same as mobile
            'password' => Hash::make($request->password),
            'is_active' => true,
        ]);

        Auth::login($user);

        Log::info("User registered: " . $user->email);

        try {
            \App\Services\UserMailService::sendWelcomeEmail($user);
        } catch (\Exception $e) {
            Log::error("Failed to send welcome email: " . $e->getMessage());
        }

        return redirect()->route('user.dashboard');
    }

    // Handle Logout
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('home');
    }

    // Google Login
    public function redirectToGoogle()
    {
        return \Laravel\Socialite\Facades\Socialite::driver('google')->redirect();
    }

    public function handleGoogleCallback()
    {
        try {
            $googleUser = \Laravel\Socialite\Facades\Socialite::driver('google')->user();

            $user = User::where('email', $googleUser->getEmail())->first();

            if (!$user) {
                // Create new user
                $user = User::create([
                    'name' => $googleUser->getName(),
                    'email' => $googleUser->getEmail(),
                    'password' => \Illuminate\Support\Facades\Hash::make(\Illuminate\Support\Str::random(24)), // Random password
                    'is_active' => true,
                    // You might want to store google_id if you add that column
                    // 'google_id' => $googleUser->getId(),
                ]);

                Log::info("User registered via Google: " . $user->email);

                try {
                    \App\Services\UserMailService::sendWelcomeEmail($user);
                } catch (\Exception $e) {
                    Log::error("Failed to send welcome email: " . $e->getMessage());
                }
            }

            Auth::login($user);
            Log::info("User logged in via Google: " . $user->email);

            return redirect()->intended(route('user.dashboard'));

        } catch (\Exception $e) {
            Log::error("Google Login Error: " . $e->getMessage());
            return redirect()->route('login')->withErrors(['email' => 'Unable to login with Google. Please try again.']);
        }
    }
}
