<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Reseller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class ResellerProfileController extends Controller
{
    /**
     * Get Reseller Profile
     */
    public function show()
    {
        $user = Auth::user();
        $reseller = Reseller::where('user_id', $user->id)->first();

        if (!$reseller) {
            return response()->json([
                'status' => 'error',
                'message' => 'Reseller profile not found.'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                ],
                'reseller' => [
                    'company_name' => $reseller->company_name,
                    'phone' => $reseller->phone,
                    'gst_number' => $reseller->gst_number,
                    'address' => $reseller->address,
                    'city' => $reseller->city,
                    'state' => $reseller->state,
                    'pincode' => $reseller->pincode,
                    'country' => $reseller->country ?? 'India',
                    'logo_url' => $reseller->logo_path ? asset('storage/' . $reseller->logo_path) : null,
                ]
            ]
        ]);
    }

    /**
     * Update Reseller Profile
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        $reseller = Reseller::where('user_id', $user->id)->firstOrFail();

        $request->validate([
            'name' => 'required|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'phone' => 'required|string|max:20',
            'gst_number' => 'nullable|string|max:20',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'pincode' => 'required|string|max:20',
            'country' => 'nullable|string|max:100',
            'logo' => 'nullable|image|max:2048',
        ]);

        // Update User Name
        $user->update(['name' => $request->name]);

        // Prepare Reseller Data
        $resellerData = $request->only([
            'company_name',
            'phone',
            'gst_number',
            'address',
            'city',
            'state',
            'pincode',
            'country'
        ]);

        // Handle Logo Upload
        if ($request->hasFile('logo')) {
            if ($reseller->logo_path) {
                Storage::disk('public')->delete($reseller->logo_path);
            }
            $resellerData['logo_path'] = $request->file('logo')->store('resellers/logos', 'public');
        }

        $reseller->update($resellerData);

        return response()->json([
            'status' => 'success',
            'message' => 'Profile updated successfully.',
            'data' => [
                'name' => $user->name,
                'reseller' => array_merge($reseller->fresh()->toArray(), [
                    'logo_url' => $reseller->logo_path ? asset('storage/' . $reseller->logo_path) : null
                ])
            ]
        ]);
    }

    /**
     * Change Password
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|current_password',
            'new_password' => ['required', 'confirmed', Password::min(8)],
        ]);

        $user = Auth::user();
        $user->update([
            'password' => Hash::make($request->new_password)
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Password updated successfully.'
        ]);
    }
}
