<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Order;
use App\Models\Product;
use App\Models\Reseller;
use App\Models\Address;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ResellerOrderController extends Controller
{
    /**
     * List Orders with Filters and Counts
     */
    public function index(Request $request)
    {
        $userId = Auth::id();
        $query = Order::where('user_id', $userId)->withCount('items');

        // Filter by Status
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $orders = $query->latest()->paginate(15);

        // Map data for mobile consumption
        $orders->getCollection()->transform(function ($order) {
            return [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'status' => $order->status,
                'status_label' => $this->getStatusLabel($order->status),
                'total' => (float) $order->total,
                'items_count' => $order->items_count,
                'created_at' => $order->created_at->format('d M Y'),
                'is_quotation' => $order->status === 'quotation_sent'
            ];
        });

        // Summary Counts (Same as web tabs)
        $counts = [
            'all' => Order::where('user_id', $userId)->count(),
            'pending' => Order::where('user_id', $userId)->where('status', 'pending')->count(),
            'quotation_received' => Order::where('user_id', $userId)->where('status', 'quotation_sent')->count(),
            'quotation_accepted' => Order::where('user_id', $userId)->where('status', 'quotation_accepted')->count(),
            'completed' => Order::where('user_id', $userId)->where('status', 'completed')->count(),
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'orders' => $orders,
                'counts' => $counts
            ]
        ]);
    }

    /**
     * Get Products for Order Creation
     */
    public function products(Request $request)
    {
        $query = Product::with(['category', 'images', 'variations'])->where('status', true);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        // Category Filter
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        $products = $query->latest()->paginate(20);

        // Format product data for mobile
        $products->getCollection()->transform(function ($product) {
            $price = $product->reseller_price ?: ($product->sp ?: $product->mrp);

            // Thumbnail logic
            $thumbnail = null;
            if ($product->images->first()) {
                $path = $product->images->first()->thumbnail_path ?: $product->images->first()->image_path;
                $thumbnail = $this->formatImageUrl($path);
            }

            return [
                'id' => $product->id,
                'name' => $product->name,
                'category' => $product->category ? $product->category->name : 'N/A',
                'mrp' => (float) $product->mrp,
                'sp' => (float) $product->sp,
                'reseller_price' => (float) $product->reseller_price,
                'final_price' => (float) $price,
                'thumbnail' => $thumbnail,
                'stock_status' => $product->stock_status ?? 'in_stock',
                'unit' => $product->unit ?? 'Unit'
            ];
        });

        $categories = Category::whereNull('parent_id')->get(['id', 'name']);

        return response()->json([
            'status' => 'success',
            'data' => [
                'products' => $products,
                'categories' => $categories
            ]
        ]);
    }

    /**
     * Place a New Order
     */
    public function store(Request $request)
    {
        $request->validate([
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        $userId = Auth::id();
        $totalAmount = 0;
        $orderItemsData = [];

        foreach ($request->items as $item) {
            $product = Product::find($item['product_id']);
            $price = $product->reseller_price ?: ($product->sp ?: $product->mrp);

            $totalAmount += $price * $item['quantity'];

            $orderItemsData[] = [
                'product_id' => $product->id,
                'price' => $price,
                'quantity' => $item['quantity'],
                'total' => $price * $item['quantity'],
            ];
        }

        try {
            $order = DB::transaction(function () use ($userId, $totalAmount, $orderItemsData) {
                $reseller = Reseller::where('user_id', $userId)->firstOrFail();

                // Create a dynamic address for this order based on reseller profile
                $address = Address::create([
                    'user_id' => $userId,
                    'name' => Auth::user()->name,
                    'phone' => $reseller->phone,
                    'address_line1' => $reseller->address,
                    'city' => $reseller->city,
                    'state' => $reseller->state,
                    'country' => $reseller->country ?? 'India',
                    'postal_code' => $reseller->pincode,
                    'type' => 'Business',
                    'is_default' => false
                ]);

                $order = Order::create([
                    'user_id' => $userId,
                    'order_number' => 'ORD-' . strtoupper(uniqid()),
                    'status' => 'pending',
                    'payment_status' => 'pending',
                    'payment_method' => 'cod', // Default for reseller orders
                    'subtotal' => $totalAmount,
                    'total' => $totalAmount,
                    'shipping_address_id' => $address->id,
                    'billing_address_id' => $address->id,
                ]);

                foreach ($orderItemsData as $itemData) {
                    $order->items()->create($itemData);
                }

                return $order;
            });

            return response()->json([
                'status' => 'success',
                'message' => 'Order placed successfully.',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to place order: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show Order Details
     */
    public function show($id)
    {
        $order = Order::with(['items.product', 'shippingAddress'])->where('user_id', Auth::id())->findOrFail($id);

        return response()->json([
            'status' => 'success',
            'data' => [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'status' => $order->status,
                'status_label' => $this->getStatusLabel($order->status),
                'created_at' => $order->created_at->format('d M Y, h:i A'),
                'subtotal' => (float) $order->subtotal,
                'shipping_cost' => (float) $order->shipping_cost,
                'discount' => (float) $order->discount,
                'total' => (float) $order->total,
                'invoice_url' => $order->status === 'quotation_sent'
                    ? route('reseller.orders.invoice', $order->id)
                    : null,
                'shipping_address' => $order->shippingAddress,
                'items' => $order->items->map(function ($item) {
                    $thumbnail = null;
                    if ($item->product && $item->product->images->first()) {
                        $path = $item->product->images->first()->thumbnail_path ?: $item->product->images->first()->image_path;
                        $thumbnail = $this->formatImageUrl($path);
                    }
                    return [
                        'id' => $item->id,
                        'product_name' => $item->product->name ?? 'Deleted Product',
                        'thumbnail' => $thumbnail,
                        'quantity' => $item->quantity,
                        'price' => (float) $item->price,
                        'total' => (float) $item->total,
                        'discount_percent' => $item->discount_percent // If applicable
                    ];
                })
            ]
        ]);
    }

    /**
     * Accept Quotation
     */
    public function acceptQuotation($id)
    {
        $order = Order::where('user_id', Auth::id())->findOrFail($id);

        if ($order->status !== 'quotation_sent') {
            return response()->json([
                'status' => 'error',
                'message' => 'This order is not pending quotation acceptance.'
            ], 422);
        }

        $order->update(['status' => 'quotation_accepted']);

        return response()->json([
            'status' => 'success',
            'message' => 'Quotation accepted successfully. The order is now processing.'
        ]);
    }

    /**
     * Format Image URL
     */
    private function formatImageUrl($path)
    {
        if (!$path)
            return null;
        if (str_starts_with($path, 'http'))
            return $path;

        // Handle public/ or storage/ prefixes
        $path = str_replace(['public/', 'storage/'], '', $path);
        return asset('storage/' . $path);
    }

    /**
     * Status Labels
     */
    private function getStatusLabel($status)
    {
        $labels = [
            'pending' => 'Pending Review',
            'quotation_sent' => 'Quotation Received',
            'quotation_accepted' => 'Quotation Accepted',
            'processing' => 'Processing',
            'shipped' => 'Shipped',
            'completed' => 'Completed',
            'cancelled' => 'Cancelled',
        ];

        return $labels[$status] ?? ucfirst($status);
    }
}
