<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ResellerDashboardController extends Controller
{
    /**
     * Reseller Dashboard Stats and Recent Activity
     */
    public function index()
    {
        $userId = Auth::id();

        // Stats
        $stats = [
            'total_orders' => Order::where('user_id', $userId)->count(),
            'pending_review' => Order::where('user_id', $userId)->where('status', 'pending')->count(),
            'quotes_received' => Order::where('user_id', $userId)->where('status', 'quotation_sent')->count(),
            'completed_orders' => Order::where('user_id', $userId)->where('status', 'completed')->count(),
        ];

        // Support Info
        $admin = User::whereHas('roles', function ($q) {
            $q->where('name', 'admin');
        })->first() ?? User::find(1);

        $support = [
            'name' => $admin->name ?? 'Support',
            'email' => $admin->email ?? 'support@lugani.com',
            'phone' => $admin->whatsapp_number ?? '+91 98765 43210',
        ];

        // Recent Activity (Last 5 orders)
        $recentOrders = Order::where('user_id', $userId)
            ->withCount('items')
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'status' => $order->status,
                    'status_label' => $this->getStatusLabel($order->status),
                    'total_amount' => (float) $order->total,
                    'items_count' => $order->items_count,
                    'created_at' => $order->created_at->format('d M Y, h:i A'),
                ];
            });

        return response()->json([
            'status' => 'success',
            'data' => [
                'stats' => $stats,
                'support' => $support,
                'recent_orders' => $recentOrders,
                'user' => [
                    'name' => Auth::user()->name,
                    'email' => Auth::user()->email,
                ]
            ]
        ]);
    }

    /**
     * Helper to get human-readable status labels
     */
    private function getStatusLabel($status)
    {
        $labels = [
            'pending' => 'Pending Review',
            'quotation_sent' => 'Quotation Received',
            'quotation_accepted' => 'Quotation Accepted',
            'processing' => 'Processing',
            'shipped' => 'Shipped',
            'completed' => 'Completed',
            'cancelled' => 'Cancelled',
        ];

        return $labels[$status] ?? ucfirst($status);
    }
}
