<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ResellerApplication;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class ResellerAuthController extends Controller
{
    /**
     * Reseller Login
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'device_name' => 'nullable|string',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid credentials.',
            ], 401);
        }

        // Check if user is active
        if (!$user->is_active) {
            return response()->json([
                'status' => 'error',
                'message' => 'Your account is inactive. Please contact support.',
            ], 403);
        }

        // Check if user has Reseller role
        if (!$user->hasRole('Reseller')) {
            return response()->json([
                'status' => 'error',
                'message' => 'This account is not registered as a reseller.',
            ], 403);
        }

        $deviceName = $request->device_name ?? 'reseller_app';
        $token = $user->createToken($deviceName)->plainTextToken;

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful',
            'data' => [
                'token' => $token,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'role' => 'Reseller'
                ]
            ]
        ]);
    }

    /**
     * Reseller Registration (Application Submission)
     */
    public function signup(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:reseller_applications,email',
            'phone' => 'required|string|max:20',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'postal_code' => 'required|string|max:20',
            'company_name' => 'required|string|max:255',
            'company_website' => 'nullable|url|max:255',
            'gst_number' => 'required|string|max:50',
        ]);

        $application = new ResellerApplication($validated);
        $application->status = 'pending';

        // If the user happens to be logged in (e.g. as a regular customer)
        if (Auth::guard('sanctum')->check()) {
            $application->user_id = Auth::guard('sanctum')->id();
        }

        $application->save();

        return response()->json([
            'status' => 'success',
            'message' => 'Your application has been submitted successfully. Our team will review it and contact you soon.',
            'data' => [
                'application_id' => $application->id
            ]
        ], 201);
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Logged out successfully'
        ]);
    }
}
